from pathlib import Path
import http.client
import asyncio
import json
import subprocess
import websockets
import os
import shutil
import zipfile
import tempfile
import requests
import sys
import psutil
import platform
import time
import ctypes

ctypes.windll.kernel32.SetConsoleTitleW("UpdateRustServerThree")
BASE_DIR = Path(__file__).resolve().parent

# Backend Port
PORT = 28035

# Script Names
RUST_SERVER_SCRIPT = "StartRustServerThree.py"
TARGET_SCRIPTS = {
    "RconServerThree.py",
    "LogStreamServerThree.py"
}

DOWNLOAD_URL = "https://rustcontrolpanel.stiffgoat.com/download"

EXCLUDED_FILES = {
    "ServerStartupArgs.py"
}

CLIENT_WS = None
UPDATER_PID = os.getpid()
ERROR_MESSAGE = ""

def finished(success, message):
    return {
        "type": "server-done",
        "success": success,
        "message": message
    }

def status(message):
    return {
        "type": "status",
        "message": message
    }

def progress(step, percent):
    return {
        "type": "progress",
        "step": step,
        "percent": percent
    }

def error(bool):
    return {
        "type": "error",
        "bool": bool
    }

#--- Function Definitions ---
async def download_update(dest_path):
    try:
        r = requests.get(DOWNLOAD_URL, stream=True, timeout=30)
        r.raise_for_status()

        with open(dest_path, "wb") as f:
            for chunk in r.iter_content(chunk_size=8192):
                if chunk:
                    f.write(chunk)

    except Exception as e:
        await broadcast(status("Can't download update! Force exit!"))
        await asyncio.sleep(3)
        raise RuntimeError(f"Download failed: {e}\n")

async def extract_zip(zip_path, extract_to):
    try:
        with zipfile.ZipFile(zip_path, "r") as z:
            z.extractall(extract_to)
    except Exception as e:
        await broadcast(status("Can't unpack update! Force exit!"))
        await asyncio.sleep(3)
        raise RuntimeError(f"Zip extraction failed: {e}.\n")

async def overlay_copy(src, dst, max_retries=2):
    backup_dir = tempfile.mkdtemp(prefix="panel_backup_")

    try:
        for root, dirs, files in os.walk(dst):
            rel_path = os.path.relpath(root, dst)
            target_dir = os.path.join(backup_dir, rel_path) if rel_path != "." else backup_dir
            os.makedirs(target_dir, exist_ok=True)

            for file in files:
                src_file = os.path.join(root, file)
                dst_file = os.path.join(target_dir, file)
                shutil.copy2(src_file, dst_file)

        for root, dirs, files in os.walk(src):
            rel_path = os.path.relpath(root, src)
            target_dir = dst if rel_path == "." else os.path.join(dst, rel_path)
            os.makedirs(target_dir, exist_ok=True)

            for file in files:
                if file in EXCLUDED_FILES:
                    continue

                src_file = os.path.join(root, file)
                dst_file = os.path.join(target_dir, file)

                for attempt in range(max_retries + 1):
                    try:
                        shutil.copy2(src_file, dst_file)
                        break
                    except Exception as e:
                        if attempt < max_retries:
                            await asyncio.sleep(0.5)
                            continue
                        else:
                            raise RuntimeError(
                                f"Failed to copy {src_file} to {dst_file} "
                                f"after {max_retries+1} attempts: {e}.\n\n"
                            )

    except Exception as e:
        if os.path.exists(dst):
            shutil.rmtree(dst)
        shutil.copytree(backup_dir, dst)
        raise RuntimeError(
            f"Overlay copy failed, rolled back to previous state. Reason: {e}.\n\n"
        )

    finally:
        if os.path.exists(backup_dir):
            shutil.rmtree(backup_dir)

async def stop_rust_server():
    await broadcast(progress("panel", 10))
    try:
        conn = http.client.HTTPConnection("127.0.0.1", 28036)
        conn.request("PUT", "/Stop_Server", headers={"X-Updater-PID": str(UPDATER_PID)})
        conn.getresponse()
        conn.close()
        print("Rust server stopped successfully.")
    except Exception as e:
        print(f"Error while killing server processes: {e}")

async def install_panel_files():
    await broadcast(progress("panel", 20))

    with tempfile.TemporaryDirectory() as tmp:
        zip_path = os.path.join(tmp, "download.zip")
        extract_path = os.path.join(tmp, "extract")

        # Download
        await broadcast(status("Downloading new Rust server files"))
        print("Downloading and installing updates")
        await download_update(zip_path)
        await broadcast(progress("panel", 35))
        await asyncio.sleep(2)

        # Extract
        await broadcast(status("Unpacking the new files"))
        print("Unpacking the new files")
        await extract_zip(zip_path, extract_path)
        await broadcast(progress("panel", 50))
        await asyncio.sleep(2)

        # Validate structure
        await broadcast(status("Verifying file structure"))
        print("Verifying file structure")
        src_panel_dir = os.path.join(
            extract_path,
            "Rust_Control_Panel",
            "RustServerThree"
        )
        await broadcast(progress("panel", 60))
        await asyncio.sleep(2)

        if not os.path.isdir(src_panel_dir):
            await broadcast(status("File structure invalid, force exit!"))
            print("File structure invalid, force exit!")
            await asyncio.sleep(3)
            raise RuntimeError("Invalid update package directory structure.\n")

        # Overlay copy into existing Rust server directory
        await broadcast(status("Copying new files to Rust server"))
        print("Copying new files to Rust server")
        dst_panel_dir = os.path.dirname(os.path.abspath(__file__))
        await overlay_copy(src_panel_dir, dst_panel_dir)
        await broadcast(progress("panel", 70))
        await asyncio.sleep(2)

async def start_rust_server():
    await broadcast(progress("panel", 80))
    try:
        conn = http.client.HTTPConnection("127.0.0.1", 28036)
        conn.request("PUT", "/Start_Server")
        conn.getresponse()
        conn.close()
        print("Rust server started successfully.")
    except Exception as e:
        print(f"Failed to start Rust server: {e}")

async def is_script_running(script_name):
    system = platform.system()
    try:
        if system == "Windows":
            cmd = ['wmic', 'process', 'get', 'CommandLine']
        else:
            cmd = ['ps', 'aux']

        output = subprocess.check_output(cmd, text=True, errors='ignore')
        return any(script_name in line and 'python' in line for line in output.splitlines())
    
    except Exception:
        return False

#--- Rust Server Update Process ---
async def run_panel_update():
    global ERROR_MESSAGE
    update_failed = False

    await broadcast(status("Rust server update started..."))
    print("Rust server update started...")
    await asyncio.sleep(3)
    await broadcast(error(False))

    # 1. Stop services
    print("Stopping backend services")
    await stop_rust_server()
    await asyncio.sleep(3)

    if await is_script_running("RconServerThree"):
        await broadcast(status("RconServerThree did not stop!"))
        print("RconServerThree did not stop!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "RconServerThree did not stop during the update.\nA manual restart of this service is required.\n\n"
    else:
        await broadcast(status("RconServerThree stopped successfully"))
        print("RconServerThree stopped successfully")
    await asyncio.sleep(2)
    
    if await is_script_running("LogStreamServerThree"):
        await broadcast(status("LogStreamServerThree did not stop!"))
        print("LogStreamServerThree did not stop!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "LogStreamServerThree did not stop during the update.\nA manual restart of this service is required.\n\n"
    else:
        await broadcast(status("LogStreamServerThree stopped successfully"))
        print("LogStreamServerThree stopped successfully")
    await asyncio.sleep(2)

    if await is_script_running("StartRustServerThree"):
        await broadcast(status("RustGameServerThree did not stop!"))
        print("RustGameServerThree did not stop!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "RustGameServerThree did not stop during the update.\nA manual restart of this service is required.\n\n"
    else:
        await broadcast(status("RustGameServerThree stopped successfully"))
        print("RustGameServerThree stopped successfully")      
    await asyncio.sleep(2)
    await broadcast(progress("panel", 15))

    # 2. Install / update files
    try:
        await install_panel_files()
        await asyncio.sleep(3)

    except Exception as e:
        update_failed = True
        ERROR_MESSAGE += "Critical Error:\n"
        ERROR_MESSAGE += f"{e}\n"
        await broadcast(error(True))

    # 3. Start services
    print("Restarting backend services...")
    await broadcast(status("Restarting backend services"))
    await start_rust_server()
    await asyncio.sleep(3)

    print("Verifying backend status...")
    await broadcast(status("Verifying backend is running"))
    await broadcast(progress("panel", 90))
    await asyncio.sleep(3)

    if not await is_script_running("StartRustServerThree"):
        await broadcast(status("RustGameServerThree did not start!"))
        print("RustGameServerThree did not start!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "RustGameServerThree did not restart after the update.\nA manual start of this service is required.\n\n"
    else:
        await broadcast(status("RustGameServerThree is running"))
        print("RustGameServerThree is running")
    await asyncio.sleep(2)
    
    if not await is_script_running("RconServerThree"):
        await broadcast(status("RconServerThree did not start!"))
        print("RconServerThree did not start!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "RconServerThree did not restart after the update.\nA manual start of this service is required.\n\n"
    else:
        await broadcast(status("RconServerThree is running"))
        print("RconServerThree is running")
    await asyncio.sleep(2)
    
    if not await is_script_running("LogStreamServerThree"):
        await broadcast(status("LogStreamServerThree did not start!"))
        print("LogStreamServerThree did not start!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "LogStreamServerThree did not restart after the update.\nA manual start of this service is required.\n\n"
    else:
        await broadcast(status("LogStreamServerThree is running"))
        print("LogStreamServerThree is running")
    await asyncio.sleep(2)
        
    if not update_failed:
        await broadcast(status("Rust server update succeeded"))
        print("Rust server update succeeded")
    else:
        await broadcast(status("Update exited early with an error"))
        print("Update exited early with an error")
    await broadcast(progress("panel", 95))
    await asyncio.sleep(3)

    if not update_failed:
        await broadcast(finished(True, ERROR_MESSAGE))
    else:
        await broadcast(finished(False, ERROR_MESSAGE))

# --- Websocket Server ---
async def broadcast(payload):
    if CLIENT_WS:
        await CLIENT_WS.send(json.dumps(payload))

async def handler(ws):
    global CLIENT_WS
    CLIENT_WS = ws

    print("Update controller connected")

    try:
        async for msg in ws:
            data = json.loads(msg)

            if data.get("type") == "command" and data.get("action") == "start":
                print("Start command received")
                await asyncio.sleep(3)
                await run_panel_update()
                await asyncio.sleep(1)

    except websockets.ConnectionClosed:
        print("Controller disconnected")

    finally:
        quit()

async def main():
    async with websockets.serve(handler, "0.0.0.0", PORT):
        print(f"UpdateRustServer listening on port {PORT}")
        await asyncio.Future()

asyncio.run(main())

