using Oxide.Core;
using System.Collections.Generic;
using UnityEngine;
using WebSocketSharp;
using WebSocketSharp.Server;
using System;
using Newtonsoft.Json;
using System.IO;

namespace Oxide.Plugins
{
    [Info("WebMinimap", "Im_just_a_Pixel", "1.2.54")]
    [Description("Streams Rust minimap image and player positions to control panel, tracks deaths")]
    public class WebMinimap : CovalencePlugin
    {
        private readonly object clientsLock = new object();
        private WebSocketServer wssv;
        private List<MapConnection> clients = new List<MapConnection>();
        private byte[] mapImageData;
        private int mapSize = 0;
        private float worldSize;
        private float scale = 1.0f;
        private int renderedWidth = 0;
        private int renderedHeight = 0;
        private PatrolHelicopterAI cachedPatrolHeli = null;

        private List<DeathEntry> deathHistory = new List<DeathEntry>();
        private string deathDataPath => $"{Interface.Oxide.DataDirectory}/WebMinimap.json";

        private Dictionary<string, Vector3> knownSleepers = new Dictionary<string, Vector3>();

        private class DeathEntry
        {
            public string id;
            public string name;
            public float x;
            public float z;
            public DateTime timestamp;
            public bool online;
        }

        private class MapConnection : WebSocketBehavior
        {
            protected override void OnOpen()
            {
                var plugin = Interface.Oxide.RootPluginManager.GetPlugin("WebMinimap") as WebMinimap;
                lock (plugin.clientsLock)
                {
                    plugin.clients.Add(this);
                }
                plugin.SendInitialData(this);
            }

            protected override void OnClose(CloseEventArgs e)
            {
                var plugin = Interface.Oxide.RootPluginManager.GetPlugin("WebMinimap") as WebMinimap;
                lock (plugin.clientsLock)
                {
                    plugin.clients.Remove(this);
                }
            }

            public void SendData(string data) => Send(data);
        }

        private void Init()
        {
            Puts("Plugin loaded, waiting for server to initialize...");
            LoadDeathData();
        }

        private void OnServerInitialized()
        {
            timer.Once(3f, () =>
            {
                Puts("Server initialized, generating map and starting tracker...");
                worldSize = TerrainMeta.Size.x;
                GenerateMapImage();

                wssv = new WebSocketServer("ws://0.0.0.0:28032");
                wssv.AddWebSocketService<MapConnection>("/map");
                wssv.Start();

                timer.Every(2f, () =>
                {
                    CheckOfflineSleeperDeaths();
                    SendPlayerPositions();
                    CleanupOldDeaths();
                });
            });
        }

        private void Unload()
        {
            if (wssv != null && wssv.IsListening)
                wssv.Stop();

            wssv = null;
            clients.Clear();
            Puts("WebSocket server stopped.");
        }

        private void GenerateMapImage()
        {
            try
            {
                Color background;
                float desiredResolution = 1500f;
                scale = desiredResolution / TerrainMeta.Size.x;

                var result = MapImageRenderer.Render(
                    out renderedWidth,
                    out renderedHeight,
                    out background,
                    scale,
                    false,
                    false,
                    mapSize
                );

                if (result is byte[] rawBytes)
                {
                    mapImageData = rawBytes;
                    Puts($"Map rendered: {renderedWidth}x{renderedHeight}, {mapImageData.Length} bytes.");
                }
                else
                {
                    PrintError("Unexpected return type from MapImageRenderer.Render()");
                }
            }
            catch (Exception ex)
            {
                PrintError($"Failed to render map: {ex.Message}");
            }
        }

        private void SendInitialData(MapConnection client)
        {
            if (mapImageData == null) return;

            float mapWorldWidth = renderedWidth / scale;
            float mapWorldHeight = renderedHeight / scale;

            var meta = new
            {
                type = "map_meta",
                worldSize = worldSize,
                mapSize = mapSize,
                image = Convert.ToBase64String(mapImageData),
                bounds = new
                {
                    left = -mapWorldWidth / 2f,
                    right = mapWorldWidth / 2f,
                    top = mapWorldHeight / 2f,
                    bottom = -mapWorldHeight / 2f
                }
            };

            client.SendData(JsonConvert.SerializeObject(meta));
        }

        private void SendPlayerPositions()
        {
            var players = GetPlayerData();
            var payload = new
            {
                type = "player_positions",
                players = players
            };

            string json = JsonConvert.SerializeObject(payload);

            lock (clientsLock)
            {
                foreach (var client in clients)
                {
                    client.SendData(json);
                }
            }
        }

        private List<object> GetPlayerData()
        {
            var players = new List<object>();
            var onlineIds = new HashSet<ulong>();
            var deadIds = new HashSet<string>(deathHistory.ConvertAll(d => d.id));

            foreach (var player in BasePlayer.activePlayerList)
            {
                if (deadIds.Contains(player.UserIDString)) continue;

                players.Add(new
                {
                    id = player.UserIDString,
                    name = player.displayName,
                    x = player.transform.position.x,
                    z = player.transform.position.z,
                    sleeping = false,
                    entity = false
                });

                onlineIds.Add(player.userID);
            }

            knownSleepers.Clear();

            foreach (var sleeper in BasePlayer.sleepingPlayerList)
            {
                if (onlineIds.Contains(sleeper.userID)) continue;
                if (deadIds.Contains(sleeper.UserIDString)) continue;

                players.Add(new
                {
                    id = sleeper.UserIDString,
                    name = sleeper.displayName,
                    x = sleeper.transform.position.x,
                    z = sleeper.transform.position.z,
                    sleeping = true,
                    entity = false
                });

                knownSleepers[sleeper.UserIDString] = sleeper.transform.position;
            }

            if (cachedPatrolHeli != null && cachedPatrolHeli.gameObject.activeInHierarchy)
            {
                var heliPos = cachedPatrolHeli.transform.position;
                players.Add(new { name = "Patrol Heli", x = heliPos.x, z = heliPos.z, entity = true });
            }

            foreach (var entity in BaseNetworkable.serverEntities)
            {
                if (entity == null || !entity.isServer) continue;

                var pos = entity.transform.position;

                if (entity is CH47HelicopterAIController)
                {
                    players.Add(new { name = "Cargo Heli", x = pos.x, z = pos.z, entity = true });
                }
                else if (entity is CargoShip)
                {
                    players.Add(new { name = "Cargo Ship", x = pos.x, z = pos.z, entity = true });
                }
                else if (entity is BradleyAPC)
                {
                    players.Add(new { name = "Bradley APC", x = pos.x, z = pos.z, entity = true });
                }
                else if (entity is CargoPlane)
                {
                    players.Add(new { name = "Cargo Plane", x = pos.x, z = pos.z, entity = true });
                }
            }

            foreach (var death in deathHistory)
            {
                players.Add(new
                {
                    id = death.id,
                    name = $"{death.name}",
                    x = death.x,
                    z = death.z,
                    dead = true,
                    online = death.online
                });
            }

            return players;
        }

        private void CheckOfflineSleeperDeaths()
        {
            var currentSleepers = new HashSet<string>();
            foreach (var s in BasePlayer.sleepingPlayerList)
                currentSleepers.Add(s.UserIDString);

            var vanished = new List<string>();
            foreach (var known in knownSleepers.Keys)
            {
                if (!currentSleepers.Contains(known) && !IsPlayerOnline(known))
                {
                    vanished.Add(known);
                }
            }

            foreach (var id in vanished)
            {
                if (deathHistory.Exists(d => d.id == id)) continue;

                var name = covalence.Players.FindPlayerById(id)?.Name ?? "Unknown";
                var pos = knownSleepers[id];

                deathHistory.Add(new DeathEntry
                {
                    id = id,
                    name = name,
                    x = pos.x,
                    z = pos.z,
                    timestamp = DateTime.UtcNow,
                    online = false
                });

                SaveDeathData();
            }
        }

        private void OnPlayerDeath(BasePlayer player, HitInfo info)
        {
            if (player == null || string.IsNullOrEmpty(player.UserIDString)) return;

            var entry = new DeathEntry
            {
                id = player.UserIDString,
                name = player.displayName,
                x = player.transform.position.x,
                z = player.transform.position.z,
                timestamp = DateTime.UtcNow,
                online = true
            };

            deathHistory.RemoveAll(d => d.id == entry.id);
            deathHistory.Add(entry);
            SaveDeathData();
        }

        private bool IsPlayerOnline(string userId)
        {
            foreach (var player in BasePlayer.activePlayerList)
                if (player.UserIDString == userId)
                    return true;
            return false;
        }

        private void OnPlayerRespawned(BasePlayer player) => RemoveDeathEntry(player);
        private void OnPlayerSleepEnded(BasePlayer player) => RemoveDeathEntry(player);

        private void RemoveDeathEntry(BasePlayer player)
        {
            bool removed = deathHistory.RemoveAll(d => d.id == player.UserIDString) > 0;
            if (removed)
                SaveDeathData();
        }

        private void SaveDeathData()
        {
            try
            {
                var json = JsonConvert.SerializeObject(deathHistory, Formatting.Indented);
                File.WriteAllText(deathDataPath, json);
            }
            catch (Exception ex)
            {
                PrintError($"Failed to save death data: {ex.Message}");
            }
        }

        private void LoadDeathData()
        {
            try
            {
                if (File.Exists(deathDataPath))
                {
                    var json = File.ReadAllText(deathDataPath);
                    deathHistory = JsonConvert.DeserializeObject<List<DeathEntry>>(json) ?? new List<DeathEntry>();
                }
            }
            catch (Exception ex)
            {
                PrintError($"Failed to load death data: {ex.Message}");
                deathHistory = new List<DeathEntry>();
            }
        }

        private void OnEntitySpawned(BaseNetworkable entity)
        {
            var baseEntity = entity as BaseEntity;
            if (baseEntity == null) return;

            var heli = baseEntity.GetComponent<PatrolHelicopterAI>();
            if (heli != null)
                cachedPatrolHeli = heli;
        }

        private void OnEntityKill(BaseNetworkable entity)
        {
            if (cachedPatrolHeli == null) return;

            var baseEntity = entity as BaseEntity;
            if (baseEntity == null) return;

            var heli = baseEntity.GetComponent<PatrolHelicopterAI>();
            if (heli != null && heli == cachedPatrolHeli)
                cachedPatrolHeli = null;
        }

        private void CleanupOldDeaths()
        {
            DateTime now = DateTime.UtcNow;

            int removed = deathHistory.RemoveAll(entry =>
            {
                TimeSpan age = now - entry.timestamp;

                if (entry.name == "Unknown" && age.TotalSeconds > 15)
                    return true;

                bool isPlayer = ulong.TryParse(entry.id, out ulong steamId) && steamId >= 76561197960265728;

                if (!isPlayer && age.TotalMinutes > 10)
                    return true;

                if (isPlayer && age.TotalHours > 48)
                    return true;

                return false;
            });

            if (removed > 0)
                SaveDeathData();
        }

        private void OnPlayerDisconnected(BasePlayer player, string reason)
        {
            var death = deathHistory.Find(d => d.id == player.UserIDString);
            if (death != null)
            {
                death.online = false;
                SaveDeathData();
            }
        }
    }
}
