using System;
using System.Collections.Generic;
using UnityEngine;
using Oxide.Core;
using Oxide.Core.Libraries.Covalence;

namespace Oxide.Plugins
{
    [Info("TeleportXYZ", "Im_just_a_Pixel", "1.1.8")]
    [Description("Adds teleport commands, persists last death positions, and allows safe chat-based teleportation.")]

    public class TeleportXYZ : RustPlugin
    {
        private const string dataFileName = "TeleportXYZ";
        private Dictionary<string, DeathData> deathLocations;
        private Dictionary<ulong, float> lastDamagedTime = new Dictionary<ulong, float>();

        // Coordinate of destination for /cod command
        private Vector3 codTeleportLocation = new Vector3(-1137.37f, 1.85f, -670.67f);

        private class DeathData
        {
            public float x, y, z;
            public long timestamp;
            public string playerName;
        }

        private void Init()
        {
            LoadDeathData();
            timer.Every(600f, CleanOldEntries); // Clean every 10 minutes
        }

        private void LoadDeathData()
        {
            deathLocations = Interface.Oxide.DataFileSystem.ReadObject<Dictionary<string, DeathData>>(dataFileName) ?? new Dictionary<string, DeathData>();
        }

        private void SaveDeathData()
        {
            Interface.Oxide.DataFileSystem.WriteObject(dataFileName, deathLocations);
        }

        private void OnPlayerDeath(BasePlayer player, HitInfo info)
        {
            if (player == null) return;

            string lowerName = player.displayName.ToLower();
            deathLocations[lowerName] = new DeathData
            {
                x = player.transform.position.x,
                y = player.transform.position.y,
                z = player.transform.position.z,
                timestamp = DateTimeToUnixTimestamp(DateTime.UtcNow),
                playerName = player.displayName
            };

            SaveDeathData();
        }

        private void CleanOldEntries()
        {
            long now = DateTimeToUnixTimestamp(DateTime.UtcNow);
            int removed = 0;

            foreach (var key in new List<string>(deathLocations.Keys))
            {
                if (now - deathLocations[key].timestamp > 172800)
                {
                    deathLocations.Remove(key);
                    removed++;
                }
            }

            if (removed > 0)
                SaveDeathData();
        }

        private long DateTimeToUnixTimestamp(DateTime time)
        {
            return (long)(time.Subtract(new DateTime(1970, 1, 1))).TotalSeconds;
        }

        private void OnEntityTakeDamage(BaseCombatEntity entity, HitInfo info)
        {
            if (entity is BasePlayer player && info?.damageTypes?.Total() > 0)
            {
                lastDamagedTime[player.userID] = Time.realtimeSinceStartup;
            }
        }

        [ChatCommand("cod")]
        private void CmdCod(BasePlayer player, string command, string[] args)
        {
            if (player == null) return;

            float currentTime = Time.realtimeSinceStartup;

            if (player.IsWounded() || player.IsSleeping() || player.IsDead())
            {
                player.ChatMessage("You cannot use /cod while wounded, sleeping, or dead.");
                return;
            }

            if (lastDamagedTime.TryGetValue(player.userID, out float lastDamage))
            {
                if (currentTime - lastDamage < 10f)
                {
                    player.ChatMessage("You cannot use /cod while in combat. Wait a few seconds.");
                    return;
                }
            }

            float safeY = TerrainMeta.HeightMap.GetHeight(new Vector3(codTeleportLocation.x, 0, codTeleportLocation.z));
            player.Teleport(new Vector3(codTeleportLocation.x, safeY + 0.2f, codTeleportLocation.z));
            player.ChatMessage("You have teleported to COD minigames lobby.");
        }

        [ConsoleCommand("teleportxyz")]
        private void TeleportXYZCommand(ConsoleSystem.Arg arg)
        {
            if (!arg.IsAdmin)
            {
                arg.ReplyWith("You do not have permission to use this command.");
                return;
            }

            if (arg.Args == null || arg.Args.Length != 4)
            {
                arg.ReplyWith("Usage: teleportxyz <playername> <x> <y> <z>");
                return;
            }

            string playerName = arg.Args[0];
            if (!float.TryParse(arg.Args[1], out float x) ||
                !float.TryParse(arg.Args[2], out float y) ||
                !float.TryParse(arg.Args[3], out float z))
            {
                arg.ReplyWith("Invalid coordinates. Must be numbers.");
                return;
            }

            BasePlayer player = FindPlayerByName(playerName);
            if (player == null)
            {
                arg.ReplyWith($"Player '{playerName}' not found.");
                return;
            }

            player.Teleport(new Vector3(x, y, z));
            arg.ReplyWith($"Teleported {player.displayName} to ({x}, {y}, {z})");
        }

        [ConsoleCommand("teleportxz")]
        private void TeleportXZCommand(ConsoleSystem.Arg arg)
        {
            if (!arg.IsAdmin)
            {
                arg.ReplyWith("You do not have permission to use this command.");
                return;
            }

            if (arg.Args == null || arg.Args.Length != 3)
            {
                arg.ReplyWith("Usage: teleportxz <playername> <x> <z>");
                return;
            }

            string playerName = arg.Args[0];
            if (!float.TryParse(arg.Args[1], out float x) ||
                !float.TryParse(arg.Args[2], out float z))
            {
                arg.ReplyWith("Invalid coordinates. Must be numbers.");
                return;
            }

            BasePlayer player = FindPlayerByName(playerName);
            if (player == null)
            {
                arg.ReplyWith($"Player '{playerName}' not found.");
                return;
            }

            float y = TerrainMeta.HeightMap.GetHeight(new Vector3(x, 0f, z));
            player.Teleport(new Vector3(x, y + 0.2f, z));
            arg.ReplyWith($"Teleported {player.displayName} to ground at ({x}, {y}, {z})");
        }

        [ConsoleCommand("teleportsleeper")]
        private void TeleportToSleeperCommand(ConsoleSystem.Arg arg)
        {
            if (!arg.IsAdmin)
            {
                arg.ReplyWith("You do not have permission to use this command.");
                return;
            }

            if (arg.Args == null || arg.Args.Length != 2)
            {
                arg.ReplyWith("Usage: teleportsleeper <playername> <sleepername>");
                return;
            }

            string playerName = arg.Args[0];
            string sleeperName = arg.Args[1];

            BasePlayer player = FindPlayerByName(playerName);
            if (player == null)
            {
                arg.ReplyWith($"Player '{playerName}' not found.");
                return;
            }

            BasePlayer sleeper = FindSleeperByName(sleeperName);
            if (sleeper == null)
            {
                arg.ReplyWith($"Sleeper '{sleeperName}' not found.");
                return;
            }

            Vector3 targetPosition = sleeper.transform.position + new Vector3(0f, 1.5f, 0f); // Avoid clipping
            player.Teleport(targetPosition);
            player.StartSleeping();
            timer.Once(5f, () => player.EndSleeping());
            arg.ReplyWith($"Teleported {player.displayName} to sleeper {sleeper.displayName} at {targetPosition}");
        }

        [ConsoleCommand("teleportdead")]
        private void TeleportToDeathCommand(ConsoleSystem.Arg arg)
        {
            if (!arg.IsAdmin)
            {
                arg.ReplyWith("You do not have permission to use this command.");
                return;
            }

            if (arg.Args == null || arg.Args.Length != 2)
            {
                arg.ReplyWith("Usage: teleportdead <playername> <targetplayername>");
                return;
            }

            string senderName = arg.Args[0];
            string targetName = arg.Args[1];

            BasePlayer sender = FindPlayerByName(senderName);
            if (sender == null)
            {
                arg.ReplyWith($"Player '{senderName}' not found.");
                return;
            }

            string targetKey = targetName.ToLower();
            DeathData data = null;

            foreach (var entry in deathLocations)
            {
                if (entry.Key.Contains(targetKey))
                {
                    data = entry.Value;
                    break;
                }
            }

            if (data == null)
            {
                if (targetName.Contains("_"))
                {
                    targetKey = targetName.Replace("_", " ").ToLower();
                    foreach (var entry in deathLocations)
                    {
                        if (entry.Key.Contains(targetKey))
                        {
                            data = entry.Value;
                            break;
                        }
                    }
                }
            }

            if (data == null)
            {
                arg.ReplyWith($"No stored death location for '{targetName}'.");
                return;
            }

            Vector3 deathPos = new Vector3(data.x, data.y + 0.2f, data.z);
            sender.Teleport(deathPos);
            sender.StartSleeping();
            timer.Once(5f, () => sender.EndSleeping());
            arg.ReplyWith($"Teleported {sender.displayName} to last death location of '{data.playerName}' at ({data.x}, {data.y}, {data.z})");
        }

        private BasePlayer FindPlayerByName(string name)
        {
            if (string.IsNullOrEmpty(name))
                return null;

            string lowered = name.ToLower();

            foreach (var player in BasePlayer.activePlayerList)
            {
                if (player.displayName.ToLower().Contains(lowered))
                    return player;
            }

            if (lowered.Contains("_"))
            {
                string spaced = lowered.Replace("_", " ");

                foreach (var player in BasePlayer.activePlayerList)
                {
                    if (player.displayName.ToLower().Contains(spaced))
                        return player;
                }
            }

            return null;
        }

        private BasePlayer FindSleeperByName(string name)
        {
            if (string.IsNullOrEmpty(name))
                return null;

            string lowered = name.ToLower();

            foreach (var sleeper in BasePlayer.sleepingPlayerList)
            {
                if (sleeper.displayName.ToLower().Contains(lowered))
                    return sleeper;
            }

            if (lowered.Contains("_"))
            {
                string spaced = lowered.Replace("_", " ");

                foreach (var sleeper in BasePlayer.sleepingPlayerList)
                {
                    if (sleeper.displayName.ToLower().Contains(spaced))
                        return sleeper;
                }
            }

            return null;
        }
    }
}