using System;
using System.Collections.Generic;
using Oxide.Core;
using UnityEngine;

namespace Oxide.Plugins
{
    [Info("PlayerRewarder", "Im_just_a_Pixel", "1.1.4")]
    [Description("Rewards players with lootbags for every hour of continuous play. Every 8th is a large lootbag.")]
    public class PlayerRewarder : RustPlugin
    {
        private class RewardInfo
        {
            public string PlayerName;
            public int TotalBagsGiven;
        }

        private Dictionary<ulong, RewardInfo> rewardData = new Dictionary<ulong, RewardInfo>();
        private Dictionary<ulong, double> loginTimes = new Dictionary<ulong, double>();
        private Dictionary<ulong, Timer> rewardTimers = new Dictionary<ulong, Timer>(); // NEW: Track player timers

        private const string dataFile = "PlayerRewarder";
        private const string mediumBag = "halloween.lootbag.medium";
        private const string largeBag = "halloween.lootbag.large";
        private const int rewardIntervalSeconds = 3600;

        #region Hooks

        private void Init()
        {
            rewardData = Interface.Oxide.DataFileSystem.ReadObject<Dictionary<ulong, RewardInfo>>(dataFile) ?? new Dictionary<ulong, RewardInfo>();

            foreach (var player in BasePlayer.activePlayerList)
            {
                timer.Once(2f, () => OnPlayerInit(player));
            }

            StartMonitoringPlayers();
        }

        private void OnServerSave() => SaveData();

        private void Unload() => SaveData();

        private void OnPlayerInit(BasePlayer player)
        {
            ulong id = player.userID;
            loginTimes[id] = Time.realtimeSinceStartup;

            if (!rewardData.ContainsKey(id))
            {
                rewardData[id] = new RewardInfo
                {
                    PlayerName = player.displayName,
                    TotalBagsGiven = 0
                };
            }

            if (rewardTimers.TryGetValue(id, out var existingTimer))
            {
                existingTimer.Destroy();
                rewardTimers.Remove(id);
            }

            ScheduleNextReward(player, rewardIntervalSeconds);
        }

        private void OnPlayerDisconnected(BasePlayer player)
        {
            loginTimes.Remove(player.userID);

            if (rewardTimers.TryGetValue(player.userID, out var t))
            {
                t.Destroy();
                rewardTimers.Remove(player.userID);
            }
        }

        #endregion

        private void ScheduleNextReward(BasePlayer player, double delaySeconds)
        {
            if (rewardTimers.TryGetValue(player.userID, out var existingTimer))
            {
                existingTimer.Destroy();
                rewardTimers.Remove(player.userID);
            }

            rewardTimers[player.userID] = timer.Once((float)delaySeconds, () =>
            {
                if (player == null || !player.IsConnected)
                    return;

                ulong id = player.userID;

                if (!rewardData.TryGetValue(id, out var info))
                    return;

                info.PlayerName = player.displayName;
                info.TotalBagsGiven++;

                string item = (info.TotalBagsGiven % 8 == 0) ? largeBag : mediumBag;
                string message = (item == largeBag)
                    ? "<color=#FFD700>You received a LARGE lootbag for 8 hours of play!</color>"
                    : "<color=#00FF00>You received a lootbag for 1 hour of play!</color>";

                player.inventory.GiveItem(ItemManager.CreateByName(item));
                SendReply(player, message);

                Puts($"{player.displayName} ({id}) received a {(item == largeBag ? "LARGE" : "MEDIUM")} lootbag. Total: {info.TotalBagsGiven}");

                SaveData();

                ScheduleNextReward(player, rewardIntervalSeconds);
            });
        }

        private void SaveData()
        {
            Interface.Oxide.DataFileSystem.WriteObject(dataFile, rewardData);
        }

        private void StartMonitoringPlayers()
        {
            timer.Every(10f, () =>
            {
                foreach (var player in BasePlayer.activePlayerList)
                {
                    if (!loginTimes.ContainsKey(player.userID))
                    {
                        Puts($"Started tracking rewards for new player: {player.displayName} ({player.userID})");
                        OnPlayerInit(player);
                    }
                }
            });
        }

        [ChatCommand("rewards")]
        private void CmdRewards(BasePlayer player, string command, string[] args)
        {
            ulong id = player.userID;

            if (rewardData.TryGetValue(id, out var info))
            {
                int large = info.TotalBagsGiven / 8;
                int medium = info.TotalBagsGiven - large;
                SendReply(player, $"<color=#aaaaaa>You have received {medium} medium and {large} large lootbags (total {info.TotalBagsGiven}).</color>");

                if (loginTimes.TryGetValue(id, out var loginTime))
                {
                    double elapsed = Time.realtimeSinceStartup - loginTime;
                    double timeLeft = rewardIntervalSeconds - (elapsed % rewardIntervalSeconds);

                    int minutes = (int)(timeLeft / 60);
                    int seconds = (int)(timeLeft % 60);

                    SendReply(player, $"<color=#00ffff>Time until next reward: {minutes}m {seconds}s</color>");
                }
                else
                {
                    SendReply(player, "<color=#ff0000>Unable to determine time to next reward.</color>");
                }
            }
            else
            {
                SendReply(player, "No reward data found for you yet.");
            }
        }
    }
}
