from pathlib import Path
import ServerStartupArgs
import sys
import os
import subprocess
import time
import zipfile
import shutil
import ctypes
import importlib
import importlib.metadata
import platform
import atexit

ctypes.windll.kernel32.SetConsoleTitleW("RustGameServerOne")
BASE_DIR = Path(__file__).resolve().parent

LOG_FILE = BASE_DIR / "Console Logs" / "logfile.log"
STEAMCMD_EXE = BASE_DIR / "steamcmd" / "steamcmd.exe"
OXIDE_ZIP_URL = "https://umod.org/games/rust/download"
OXIDE_ZIP_PATH = BASE_DIR / "oxide.rust.zip"
OXIDE_EXTRACT_PATH = BASE_DIR / "Server_Data"
RUST_DEDICATED_EXE = BASE_DIR / "Server_Data" / "RustDedicated.exe"
LOG_STREAM_SERVER_SCRIPT = "LogStreamServerOne.py"
RCON_SERVER_SCRIPT = "RconServerOne.py"

mutex_name = "Global\\RustGameServerOne_SingleInstance"
kernel32 = ctypes.WinDLL("kernel32", use_last_error=True)
mutex = kernel32.CreateMutexW(None, False, mutex_name)
ERROR_ALREADY_EXISTS = 183

if kernel32.GetLastError() == ERROR_ALREADY_EXISTS:
    print("Another instance is already running. Exiting in 5 seconds.")
    time.sleep(5)
    sys.exit(0)

def install_package(package_name, exact_version=None):
    try:
        installed_version = importlib.metadata.version(package_name)
        
        if exact_version:
            if installed_version == exact_version:
                return True
            print(f"[Dependency] {package_name} {installed_version} found (expected {exact_version}) — fixing...")
        else:
            return True

    except importlib.metadata.PackageNotFoundError:
        if exact_version:
            print(f"[Dependency] {package_name} not found — installing {exact_version}")
        else:
            print(f"[Dependency] {package_name} not found — installing latest version")

    cmd = [sys.executable, "-m", "pip", "install"]
    if exact_version:
        cmd.append(f"{package_name}=={exact_version}")
    else:
        cmd.append(package_name)

    subprocess.check_call(cmd)
    return True

install_package("psutil", "7.1.3")
install_package("requests", "2.32.5")
install_package("websockets", "15.0.1")
install_package("rcon", "2.1.1")
install_package("urllib3", "2.6.2")
install_package("charset-normalizer", "3.4.4")
install_package("certifi")
install_package("idna")

import requests
import psutil

def delete_log_file():
    try:
        os.remove(LOG_FILE)
        print(f"Deleted log file: {LOG_FILE}")
    except FileNotFoundError:
        pass
    except PermissionError:
        try:
            # Try to rename instead
            renamed = LOG_FILE + ".old"
            os.rename(LOG_FILE, renamed)
            print(f"Renamed log file instead of deleting: {renamed}")
        except Exception as e:
            print(f"❌ Could not delete or rename log file: {e}")

def create_empty_log():
    open(LOG_FILE, "w").close()

def is_process_running(script_name):
    for proc in psutil.process_iter(['cmdline']):
        cmdline = proc.info['cmdline']
        if cmdline and any(script_name.lower() in arg.lower() for arg in cmdline):
            return True
    return False

def start_python_script(script_name):
    if platform.system() == "Windows":
        si = subprocess.STARTUPINFO()
        si.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        si.wShowWindow = 2

        subprocess.Popen(
            [sys.executable, script_name],
            startupinfo=si,
            creationflags=subprocess.CREATE_NEW_CONSOLE
        )
    else:
        subprocess.Popen([sys.executable, script_name])
        
def update_steamcmd():
    print("Updating Rust server via steamcmd...")
    with open(LOG_FILE, "a", encoding="utf-8") as log:
        log.write("🔄 Running SteamCMD update...\n")
        log.flush()
        process = subprocess.Popen([
            STEAMCMD_EXE,
            "+force_install_dir", BASE_DIR / "Server_Data",
            "+login", "anonymous",
            "+app_update", "258550",
            "+quit"
        ], stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True)

        for line in process.stdout:
            print(line, end='')  # Show in console
            log.write(line)
            log.flush()

        process.wait()
        log.write("✅ SteamCMD update complete.\n")

def download_and_extract_oxide():
    print("Downloading and extracting Oxide...")
    with open(LOG_FILE, "a", encoding="utf-8") as log:
        log.write(f"🌐 Downloading Oxide from: {OXIDE_ZIP_URL}\n")
        log.flush()
        try:
            headers = {"User-Agent": "Mozilla/5.0"}

            with requests.get(OXIDE_ZIP_URL, headers=headers, stream=True, verify=False) as r:
                r.raise_for_status()
                total_size = int(r.headers.get('content-length', 0))
                downloaded = 0
                chunk_size = 8192
                last_logged_percent = -10  # Start below 0 to ensure 0% logs if desired

                with open(OXIDE_ZIP_PATH, 'wb') as f:
                    for chunk in r.iter_content(chunk_size=chunk_size):
                        if chunk:
                            f.write(chunk)
                            downloaded += len(chunk)
                            if total_size:
                                percent = int((downloaded / total_size) * 100)
                                # Only log every 10% change
                                if percent // 10 != last_logged_percent // 10:
                                    log.write(f"Downloading: {percent}% ({downloaded // 1024} KB)\n")
                                    log.flush()
                                    print(f"Downloading: {percent}%", end="\r")
                                    last_logged_percent = percent

            log.write("✅ Oxide downloaded successfully.\n")
            log.flush()

            log.write("📦 Extracting Oxide with PowerShell...\n")
            powershell_cmd = [
                "powershell",
                "-Command",
                f"Expand-Archive -Path '{OXIDE_ZIP_PATH}' -DestinationPath '{OXIDE_EXTRACT_PATH}' -Force"
            ]

            process = subprocess.Popen(powershell_cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True)
            for line in process.stdout:
                print(line, end='')
                log.write(line)
            process.wait()

            os.remove(OXIDE_ZIP_PATH)
            log.write("✅ Oxide extracted and cleaned up.\n")

        except Exception as e:
            log.write(f"❌ Oxide setup failed: {str(e)}\n")
            print(f"❌ Oxide setup failed: {e}")

def fetch_map_name():
    importlib.reload(ServerStartupArgs)

    print("Fetching map name...")
    with open(LOG_FILE, "a", encoding="utf-8") as log:
        log.write("🧭 Fetching map data...\n")
        log.flush()

    try:
        _, map_info = ServerStartupArgs.get_custom_args()
        args, _ = ServerStartupArgs.get_custom_args()

        if map_info["map_mode"] == "seed":
            seed = map_info["seed"]
            worldsize = map_info["worldsize"]
            msg = f"🗺 Using procedural map with seed: {seed}, world size: {worldsize}\n"
            time.sleep(2)
            print(msg.strip())
            with open(LOG_FILE, "a", encoding="utf-8") as log:
                log.write(msg)
                log.flush()
            return None

        elif map_info["map_mode"] == "url":
            map_url = args[args.index("+server.levelurl") + 1]
            map_name = os.path.basename(map_url)
            time.sleep(2)
            print(f"🗺 Using custom map: {map_name}")
            print(f"🌐 Map URL: {map_url}")
            with open(LOG_FILE, "a", encoding="utf-8") as log:
                log.write(f"🗺 Using custom map: {map_name}\n")
                log.write(f"🌐 Map URL: {map_url}\n")
                log.flush()

            return map_name

        else:
            raise Exception("❌ Invalid map configuration: neither seed nor URL specified.")

    except Exception as e:
        print(f"❌ Failed to determine map config: {e}")
        with open(LOG_FILE, "a", encoding="utf-8") as log:
            log.write(f"❌ Failed to fetch map name or detect map mode: {e}\n")
        raise

def start_rust_server(map_name):
    print("Starting Rust server...")

    base_command = [
        RUST_DEDICATED_EXE,
        "-batchmode",
        "-logfile", "../Console Logs/logfile.log",
        "+server.identity", "RustServerOne",
        "+rcon.port", "28014",
        "+rcon.web", "0"
    ]
    
    args, _ = ServerStartupArgs.get_custom_args()
    rust_command = base_command + args

    subprocess.run(rust_command, cwd=str(BASE_DIR / "Server_Data"))

def main_loop():
    server_restarting = False
    while True:
        try:
            delete_log_file()
            create_empty_log()

            if not is_process_running(LOG_STREAM_SERVER_SCRIPT):
                start_python_script(LOG_STREAM_SERVER_SCRIPT)
            else:
                print("Log_Stream_Server is already running. Skipping launch...")

            if not is_process_running(RCON_SERVER_SCRIPT):
                start_python_script(RCON_SERVER_SCRIPT)
                with open(LOG_FILE, "a", encoding="utf-8") as log:
                    log.write("Rcon_Server is not running. Launching...\n")
            else:
                print("Rcon_Server is already running. Skipping launch...")
                with open(LOG_FILE, "a", encoding="utf-8") as log:
                    log.write("Rcon_Server is already running. Skipping launching...\n")

            time.sleep(3)
            update_steamcmd()
            time.sleep(3)
            download_and_extract_oxide()
            time.sleep(3)
            map_name = fetch_map_name()
            time.sleep(5)
            
            with open(LOG_FILE, "a", encoding="utf-8") as log:
                log.write("🚀 Starting Rust Server...\n")
            time.sleep(3)
            start_rust_server(map_name)

            server_restarting = True
            print("Restarting Server...")
            with open(LOG_FILE, "a", encoding="utf-8") as log:
                log.write("🔄 Restarting server...\n")

        except Exception as e:
            print(f"❌ Error occurred: {e}")

        time.sleep(1)

if __name__ == "__main__":
    main_loop()
    
