using Oxide.Core;
using System.Collections.Generic;
using UnityEngine;
using WebSocketSharp;
using WebSocketSharp.Server;
using System;
using Newtonsoft.Json;

namespace Oxide.Plugins
{
    [Info("ServerInfo", "Im_just_a_Pixel", "1.0.62")]
    [Description("Streams server FPS, in-game time, elapsed time since last events, online player status over WebSocket")]
    public class ServerInfo : CovalencePlugin
    {
        private WebSocketServer wss;
        private Dictionary<ulong, OnlinePlayerInfo> onlinePlayers = new Dictionary<ulong, OnlinePlayerInfo>();
        private Dictionary<string, double> lastEventTimes = new Dictionary<string, double>
        {
            { "patrolheli", -1 },
            { "ch47", -1 },
            { "cargoship", -1 },
            { "supplydrop", -1 }
        };

        private void OnServerInitialized()
        {
            LoadEventData();
            CacheOnlinePlayers();
            StartWebSocket();
            timer.Every(1f, BroadcastServerStats);
        }

        private void Unload()
        {
            StopWebSocket();
            SaveEventData();
        }

        private void StartWebSocket()
        {
            try
            {
                SimpleWebSocketBehavior.plugin = this;
                wss = new WebSocketServer(System.Net.IPAddress.Any, 28013);
                wss.AddWebSocketService<SimpleWebSocketBehavior>("/");
                wss.Start();
                LogInfo("Info polling server started on Port: 28013");
            }
            catch (Exception ex)
            {
                PrintError($"Failed to start WebSocket server: {ex.Message}");
            }
        }

        private void StopWebSocket()
        {
            if (wss != null)
            {
                wss.Stop();
                wss = null;
                LogInfo("ServerInfo WebSocket server stopped.");
            }
        }

        private void SaveEventData()
        {
            Interface.Oxide.DataFileSystem.WriteObject("ServerInfo", lastEventTimes);
        }

        private void LoadEventData()
        {
            try
            {
                var loaded = Interface.Oxide.DataFileSystem.ReadObject<Dictionary<string, double>>("ServerInfo");
                if (loaded != null)
                    lastEventTimes = loaded;
            }
            catch
            {
                LogInfo("No existing event data file found, starting fresh.");
            }

            string[] keys = { "patrolheli", "ch47", "cargoship", "supplydrop" };
            foreach (var key in keys)
            {
                if (!lastEventTimes.ContainsKey(key))
                    lastEventTimes[key] = -1;
            }
        }

        private void CacheOnlinePlayers()
        {
            onlinePlayers.Clear();

            foreach (var player in BasePlayer.activePlayerList)
            {
                onlinePlayers[player.userID] = new OnlinePlayerInfo
                {
                    steamid = player.userID,
                    name = player.displayName,
                    ip = StripPort(player.net?.connection?.ipaddress)
                };
            }
        }

        private void OnPlayerConnected(BasePlayer player)
        {
            onlinePlayers[player.userID] = new OnlinePlayerInfo
            {
                steamid = player.userID,
                name = player.displayName,
                ip = StripPort(player.net?.connection?.ipaddress)
            };
        }

        private void OnPlayerDisconnected(BasePlayer player, string reason)
        {
	    if (onlinePlayers.TryGetValue(player.userID, out var info))
	    {
	        info.disconnecting = true;
	    }
        }

        private string StripPort(string ipAddress)
        {
            if (string.IsNullOrEmpty(ipAddress))
                return "unknown";

            int index = ipAddress.LastIndexOf(':');
            return index > -1 ? ipAddress.Substring(0, index) : ipAddress;
        }

	private void CleanupDisconnectedPlayers()
	{
	    var removeList = new List<ulong>();
	
	    foreach (var kvp in onlinePlayers)
	    {
	        if (kvp.Value.disconnecting)
	            removeList.Add(kvp.Key);
	    }
	
	    foreach (var id in removeList)
	    {
	        onlinePlayers.Remove(id);
	    }
	}

        private void OnEntitySpawned(BaseNetworkable entity)
        {
            double serverTime = Time.realtimeSinceStartup;
            bool updated = false;

            if (entity is PatrolHelicopter || entity.GetType().Name == "PatrolHelicopterAIController")
            {
                lastEventTimes["patrolheli"] = serverTime;
                updated = true;
            }

            if (entity is CH47HelicopterAIController)
            {
                lastEventTimes["ch47"] = serverTime;
                updated = true;
            }

            if (entity is CargoShip)
            {
                lastEventTimes["cargoship"] = serverTime;
                updated = true;
            }

            if (entity is SupplyDrop)
            {
                lastEventTimes["supplydrop"] = serverTime;
                updated = true;
            }

            if (updated)
                SaveEventData(); // persist immediately
        }

        private void BroadcastServerStats()
        {
            if (wss == null || !wss.IsListening) return;

            double now = Time.realtimeSinceStartup;
            float gameHour = TOD_Sky.Instance?.Cycle.Hour ?? -1f;

            // Safe FPS access
            double fpsVal = (double)Performance.current.frameRate;

            var payload = new Dictionary<string, object>
            {
                ["fps"] = Math.Round(fpsVal, 1),
                ["game_time"] = gameHour,
                ["since_patrolheli"] = lastEventTimes["patrolheli"] < 0 ? -1 : Math.Round(now - lastEventTimes["patrolheli"]),
                ["since_ch47"] = lastEventTimes["ch47"] < 0 ? -1 : Math.Round(now - lastEventTimes["ch47"]),
                ["since_cargoship"] = lastEventTimes["cargoship"] < 0 ? -1 : Math.Round(now - lastEventTimes["cargoship"]),
                ["since_supplydrop"] = lastEventTimes["supplydrop"] < 0 ? -1 : Math.Round(now - lastEventTimes["supplydrop"]),
                ["players"] = onlinePlayers.Values
            };

            string json = JsonConvert.SerializeObject(payload);
            wss.WebSocketServices["/"].Sessions.Broadcast(json);
            
            CleanupDisconnectedPlayers();
        }

        public void LogInfo(string message)
        {
            Puts(message);
        }
    }

    public class OnlinePlayerInfo
    {
        public ulong steamid;
        public string name;
        public string ip;
        public bool disconnecting;
    }

    public class SimpleWebSocketBehavior : WebSocketBehavior
    {
        public static ServerInfo plugin;

    }
}