from datetime import datetime
from pathlib import Path
import asyncio
import websockets
import os
import ctypes
import time
import re
import psutil
import json

ctypes.windll.kernel32.SetConsoleTitleW("LogStreamServerFour")
BASE_DIR = Path(__file__).resolve().parent

LOG_FILE = BASE_DIR / "Console Logs" / "logfile.log"
SERVER_PORT = 28041

IGNORE_PREFIXES = [
    "Error sending rcon reply",
]
IGNORE_PATTERNS = [
    re.compile(r"\[RCON\]\[127\.0\.0\.1:\d+\] Disconnected: Exception"),
    re.compile(r"at Facepunch\.RCon\+RConClient\.Reply"),
    re.compile(r"\[RCON\] Auth: 127\.0\.0\.1:"),
]
STATUS_TRIGGER_PATTERN = re.compile(r"\[RCON\]\[127\.0\.0\.1:\d+\]\s+STATUS", re.IGNORECASE)

last_counters = psutil.net_io_counters()
last_time = time.time()

time.sleep(3)

async def system_stats(websocket):
    global last_counters, last_time

    while True:
        mem = psutil.virtual_memory()

        counters = psutil.net_io_counters()
        now = time.time()
        elapsed = now - last_time

        sent_per_sec = (counters.bytes_sent - last_counters.bytes_sent) / elapsed
        recv_per_sec = (counters.bytes_recv - last_counters.bytes_recv) / elapsed
        
        last_counters = counters
        last_time = now

        cpu_usage = psutil.cpu_percent(interval=None, percpu=False)

        data = {
            "cpu_usage": cpu_usage,
            "memory_usage": mem.percent,
            "memory_used": mem.used,
            "memory_total": mem.total,
            "net_sent": sent_per_sec,
            "net_recv": recv_per_sec
        }

        await websocket.send(json.dumps(data))
        await asyncio.sleep(1)

async def log_stream(websocket):
    last_size = 0
    f = None
    suppress_next_lines = 0

    try:
        while True:
            try:
                if not os.path.exists(LOG_FILE):
                    await asyncio.sleep(1)
                    continue

                current_size = os.path.getsize(LOG_FILE)

                if f is None or current_size < last_size:
                    if f:
                        f.close()
                    f = open(LOG_FILE, 'r', encoding='utf-8', errors='ignore')
                    f.seek(0, os.SEEK_END)
                    print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] 📂 Log file opened for streaming.")
                
                last_size = current_size

                line = f.readline()
                if line:
                    # Match [RCON][127.0.0.1:*] STATUS (case-insensitive)
                    if STATUS_TRIGGER_PATTERN.search(line):
                        suppress_next_lines = 5
                        continue

                    # Skip the next 5 lines after STATUS match
                    if suppress_next_lines > 0:
                        suppress_next_lines -= 1
                        continue

                    # Ignore junk log patterns
                    if should_ignore(line):
                        continue
                    
                    timestamp = datetime.now().strftime("[%Y-%m-%d %H:%M:%S]")
                    print(f"{timestamp} New Log Streamed: {line.strip()}")
                    await websocket.send(line)
                else:
                    await asyncio.sleep(0.1)

            except FileNotFoundError:
                if f:
                    f.close()
                    f = None
                last_size = 0
                await asyncio.sleep(1)

    except Exception as e:
        await websocket.send(f"⚠️ Log stream error: {str(e)}")
        if f:
            f.close()

def should_ignore(line: str) -> bool:
    if any(line.startswith(prefix) for prefix in IGNORE_PREFIXES):
        return True
    if any(pattern.search(line) for pattern in IGNORE_PATTERNS):
        return True
    return False

async def handler(connection):
    try:
        path = connection.request.path
        if path == "/info":
            await system_stats(connection)
        else:
            await log_stream(connection)
    except Exception as e:
        print(f"❌ Handler error: {e}")
        await websocket.send("Connection closed due to error.")
        await websocket.close()

async def main():
    async with websockets.serve(handler, "", SERVER_PORT):
        print(f"Log_Stream_Server started on port:{SERVER_PORT}...")
        await asyncio.Future()

if __name__ == "__main__":
    asyncio.run(main())
