import asyncio
import json
import subprocess
import websockets
import os
import shutil
import zipfile
import tempfile
import requests
import sys
import psutil
import platform
import time
import config
import ctypes
import atexit

ctypes.windll.kernel32.SetConsoleTitleW("UpdateControlPanel")

# Backend Port
PORT = 28003

DOWNLOAD_URL = "https://rustcontrolpanel.stiffgoat.com/download"

EXCLUDED_FILES = {
    "config.js",
    "config.py",
    "panelusers.json",
    "messages.txt",
    "rustserverone.log",
    "rustservertwo.log",
    "rustserverthree.log",
    "rustserverfour.log",
    "historyserverone.json",
    "historyservertwo.json",
    "historyserverthree.json",
    "historyserverfour.json"
}

FRONTEND_CLIENTS = set()
WORKERS = {}
server_success = False

UPDATE_STATE = {
    "panel": 0,
    "rust": 0
}

ERROR_MESSAGE = ""

# Rust server updater PORT config
SERVERONE_UPDATE_PORT = 28015
SERVERTWO_UPDATE_PORT = 28025
SERVERTHREE_UPDATE_PORT = 28035
SERVERFOUR_UPDATE_PORT = 28045

UPDATE_PORTS = {
    "Server_One": SERVERONE_UPDATE_PORT,
    "Server_Two": SERVERTWO_UPDATE_PORT,
    "Server_Three": SERVERTHREE_UPDATE_PORT,
    "Server_Four": SERVERFOUR_UPDATE_PORT
}

mutex_name = "Global\\UpdateControlPanel_SingleInstance"
kernel32 = ctypes.WinDLL("kernel32", use_last_error=True)
mutex = kernel32.CreateMutexW(None, False, mutex_name)
ERROR_ALREADY_EXISTS = 183

if kernel32.GetLastError() == ERROR_ALREADY_EXISTS:
    print("Another instance is already running. Exiting in 5 seconds.")
    time.sleep(5)
    sys.exit(0)

def cmd(action):
    return {
        "type": "command",
        "action": action
    }

def label(message):
    return {
        "type": "label",
        "message": message
    }

def status(message):
    return {
        "type": "status",
        "message": message
    }

def progress(step, percent):
    return {
        "type": "progress",
        "step": step,
        "percent": percent
    }

def error(bool):
    return {
        "type": "error",
        "bool": bool
    }

async def clear_console():
    os.system('cls' if os.name == 'nt' else 'clear')

#--- Function Definitions ---
async def download_update(dest_path):
    try:
        r = requests.get(DOWNLOAD_URL, stream=True, timeout=30)
        r.raise_for_status()

        with open(dest_path, "wb") as f:
            for chunk in r.iter_content(chunk_size=8192):
                if chunk:
                    f.write(chunk)

    except Exception as e:
        await broadcast(status("Can't download update! Force exit!"))
        await asyncio.sleep(3)
        raise RuntimeError(f"Download failed: {e}\n")

async def extract_zip(zip_path, extract_to):
    try:
        with zipfile.ZipFile(zip_path, "r") as z:
            z.extractall(extract_to)
    except Exception as e:
        await broadcast(status("Can't unpack update! Force exit!"))
        await asyncio.sleep(3)
        raise RuntimeError(f"Zip extraction failed: {e}.\n")

async def overlay_copy(src, dst, max_retries=2):
    backup_dir = tempfile.mkdtemp(prefix="panel_backup_")

    try:
        for root, dirs, files in os.walk(dst):
            rel_path = os.path.relpath(root, dst)
            target_dir = os.path.join(backup_dir, rel_path) if rel_path != "." else backup_dir
            os.makedirs(target_dir, exist_ok=True)

            for file in files:
                src_file = os.path.join(root, file)
                dst_file = os.path.join(target_dir, file)
                shutil.copy2(src_file, dst_file)

        for root, dirs, files in os.walk(src):
            rel_path = os.path.relpath(root, src)
            target_dir = dst if rel_path == "." else os.path.join(dst, rel_path)
            os.makedirs(target_dir, exist_ok=True)

            for file in files:
                if file in EXCLUDED_FILES:
                    continue

                src_file = os.path.join(root, file)
                dst_file = os.path.join(target_dir, file)

                for attempt in range(max_retries + 1):
                    try:
                        shutil.copy2(src_file, dst_file)
                        break
                    except Exception as e:
                        if attempt < max_retries:
                            await asyncio.sleep(0.5)
                            continue
                        else:
                            raise RuntimeError(
                                f"Failed to copy {src_file} to {dst_file} "
                                f"after {max_retries+1} attempts: {e}.\n\n"
                            )

    except Exception as e:
        if os.path.exists(dst):
            shutil.rmtree(dst)
        shutil.copytree(backup_dir, dst)
        raise RuntimeError(
            f"Overlay copy failed, rolled back to previous state. Reason: {e}.\n\n"
        )

    finally:
        if os.path.exists(backup_dir):
            shutil.rmtree(backup_dir)

async def stop_panel_services():
    TARGET_SCRIPTS = {
        "BackendListener.py",
        "BackendLogs.py"
    }

    killed = []

    await broadcast(progress("panel", 10))

    for proc in psutil.process_iter(['pid', 'name', 'cmdline']):
        try:
            cmdline = proc.info['cmdline']
            if not cmdline:
                continue

            for arg in cmdline:
                base = os.path.basename(arg)
                if base in TARGET_SCRIPTS:
                    print(f"Killing {base} (PID {proc.pid})")
                    proc.terminate()
                    proc.wait(timeout=5)
                    killed.append(base)

                    await broadcast(status(f"Stopping {base} service"))
                    await asyncio.sleep(2)
                    break

        except (psutil.NoSuchProcess, psutil.AccessDenied):
            await broadcast(status(f"{base} service could not be stopped. It's a tank."))
            continue

async def install_panel_files():
    await broadcast(progress("panel", 20))

    with tempfile.TemporaryDirectory() as tmp:
        zip_path = os.path.join(tmp, "download.zip")
        extract_path = os.path.join(tmp, "extract")

        # Download
        await broadcast(status("Downloading new control panel files"))
        print("Downloading and installing updates")
        await download_update(zip_path)
        await broadcast(progress("panel", 35))
        await asyncio.sleep(2)

        # Extract
        await broadcast(status("Unpacking the new files"))
        print("Unpacking the new files")
        await extract_zip(zip_path, extract_path)
        await broadcast(progress("panel", 50))
        await asyncio.sleep(2)

        # Validate structure
        await broadcast(status("Verifying file structure"))
        print("Verifying file structure")
        src_panel_dir = os.path.join(
            extract_path,
            "Rust_Control_Panel",
            "ControlPanel"
        )
        await broadcast(progress("panel", 60))
        await asyncio.sleep(2)

        if not os.path.isdir(src_panel_dir):
            await broadcast(status("File structure invalid, force exit!"))
            print("File structure invalid, force exit!")
            await asyncio.sleep(3)
            raise RuntimeError("Invalid update package directory structure.\n\n")

        # Overlay copy into existing ControlPanel directory
        await broadcast(status("Copying new files to control panel"))
        print("Copying new files to control panel")
        dst_panel_dir = os.path.dirname(os.path.abspath(__file__))
        await overlay_copy(src_panel_dir, dst_panel_dir)
        await broadcast(progress("panel", 70))
        await asyncio.sleep(2)

async def start_panel_services(script_name):
    await broadcast(progress("panel", 80))
    if platform.system() == "Windows":
        si = subprocess.STARTUPINFO()
        si.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        si.wShowWindow = 2

        subprocess.Popen(
            [sys.executable, script_name],
            startupinfo=si,
            creationflags=subprocess.CREATE_NEW_CONSOLE
        )
        
    else:
        subprocess.Popen([sys.executable, script_name])

async def is_script_running(script_name):
    system = platform.system()
    try:
        if system == "Windows":
            cmd = ['wmic', 'process', 'get', 'CommandLine']
        else:
            cmd = ['ps', 'aux']

        output = subprocess.check_output(cmd, text=True, errors='ignore')
        return any(script_name in line and 'python' in line for line in output.splitlines())
    
    except Exception:
        return False


#--- Control Panel Update Process ---
async def run_panel_update():
    global ERROR_MESSAGE
    update_failed = False

    await broadcast(label("Control Panel"))
    await broadcast(progress("panel", 0))

    await broadcast(status("Control panel update started..."))
    print("Control panel update started...")
    await asyncio.sleep(3)

    # 1. Stop services
    print("Stopping backend services...")
    await stop_panel_services()
    await asyncio.sleep(3)

    if await is_script_running("BackendListener"):
        await broadcast(status("BackendListener did not stop!"))
        print("BackendListener did not stop!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "BackendListener did not stop during the update.\nA manual restart of this service is required.\n\n"
        await asyncio.sleep(3)
    
    if await is_script_running("BackendLogs"):
        await broadcast(status("BackendLogs did not stop!"))
        print("BackendLogs did not stop!")
        await asyncio.sleep(3)
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "BackendLogs did not stop during the update.\nA manual restart of this service is required.\n\n"
    await broadcast(progress("panel", 15))

    # 2. Install / update files
    try:
        await install_panel_files()
        await asyncio.sleep(3)

    except Exception as e:
        update_failed = True
        ERROR_MESSAGE += "Critical Error:\n"
        ERROR_MESSAGE += f"{e}\n"
        await broadcast(error(True))

    # 3. Start services
    print("Restarting backend services...")
    await broadcast(status("Restarting backend services"))
    await start_panel_services("BackendListener.py")
    await start_panel_services("BackendLogs.py")
    await asyncio.sleep(3)

    print("Verifying backend status...")
    await broadcast(status("Verifying backend is running"))
    await broadcast(progress("panel", 90))
    await asyncio.sleep(3)
    
    if not await is_script_running("BackendListener"):
        await broadcast(status("BackendListener did not start!"))
        print("BackendListener did not start!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "BackendListener did not restart after the update.\nA manual start of this service is required.\n\n"
    else:
        await broadcast(status("BackendListener is running"))
        print("BackendListener is running")
    await asyncio.sleep(3)
    
    if not await is_script_running("BackendLogs"):
        await broadcast(status("BackendLogs did not start!"))
        print("BackendLogs did not start!")
        ERROR_MESSAGE += "Non-Critical Error:\n"
        ERROR_MESSAGE += "BackendLogs did not restart after the update.\nA manual start of this service is required.\n\n"
    else:
        await broadcast(status("BackendLogs is running"))
        print("BackendLogs is running")
    await asyncio.sleep(3)
        
    if not update_failed:
        await broadcast(status("Control panel update succeeded"))
        print("Control panel update succeeded")
    else:
        await broadcast(status("Update exited early with an error"))
        print("Update exited early with an error")
    await broadcast(progress("panel", 95))
    await asyncio.sleep(3)

    if not update_failed:
        await broadcast(progress("panel", 100))
        await broadcast(status("Update finished!"))
        await asyncio.sleep(3)
        if ERROR_MESSAGE == "":
            await broadcast({"type": "done", "success": "true", "message": ERROR_MESSAGE})
        else:
            await broadcast({"type": "done", "success": "soft_fail", "message": ERROR_MESSAGE})
    else:
        await broadcast({"type": "done", "success": "false", "message": ERROR_MESSAGE})

#--- Rust Server Update Process ---
async def run_rust_server_update(ws_connections):
    global ERROR_MESSAGE
    global server_success
    
    for key, ws in ws_connections.items():
        server = config.servers[key]
        name = server["name"]
        await broadcast(label(name))
        await broadcast(progress("panel", 0))
        await broadcast(status(f"Starting update on {name}"))

        await rust_update_id(key, ws, name)

        await broadcast(progress("panel", 100))
        await asyncio.sleep(2)

    if server_success:
        print("No Critical Errors")
        
    if not server_success:
        print("Yes Critical Errors")
        await broadcast({"type": "done", "success": "false", "message": ERROR_MESSAGE})
        await broadcast(error(True))
        raise RuntimeError("One or more Rust servers failed to complete updates without critical errors.")
        
async def rust_update_id(server_key, ws, name):
    global ERROR_MESSAGE
    global server_success
    
    await ws.send(json.dumps({
        "type": "command",
        "action": "start"
    }))

    try:
        async for msg in ws:
            data = json.loads(msg)
            await broadcast(data)

            if data.get("type") == "server-done":
                poop = data.get("success")
                print(poop)
                if data.get("success"):
                    server_success = True
                    msg = data.get("message")
                    if msg:
                        ERROR_MESSAGE += f"Server ({name}):\n\n"
                        ERROR_MESSAGE += msg
                    return
                
                if not data.get("success"):
                    server_success = False
                    msg = data.get("message")
                    if msg:
                        ERROR_MESSAGE += f"Server ({name}):\n\n"
                        ERROR_MESSAGE += msg
                    return

    except Exception as e:
#        await broadcast({
#            "type": "error",
#            "server": server_key,
#            "message": f"Connection lost: {e}"
#        })
        return False

#--- WebSocket Server ---
async def ws_handler(ws):
    hello = json.loads(await ws.recv())

    if hello["client"] == "frontend":
        FRONTEND_CLIENTS.add(ws)
        print("[WS] Frontend connected")

    elif hello["client"] == "rust-updater":
        WORKERS["rust"] = ws
        print("[WS] Rust updater connected")

    try:
        async for msg in ws:
            data = json.loads(msg)
            await handle_message(ws, data)

    finally:
        FRONTEND_CLIENTS.discard(ws)
        if WORKERS.get("rust") == ws:
            del WORKERS["rust"]


async def handle_message(ws, data):
    if data["type"] == "progress":
        UPDATE_STATE[data["step"]] = data["percent"]
        await broadcast_progress()

    elif data["type"] in ("status", "error"):
        await broadcast(data)


async def broadcast(msg):
    dead = set()
    for ws in FRONTEND_CLIENTS:
        try:
            await ws.send(json.dumps(msg))
        except:
            dead.add(ws)
    FRONTEND_CLIENTS.difference_update(dead)

async def wait_for_frontend(timeout=20):
    start = asyncio.get_event_loop().time()

    while not FRONTEND_CLIENTS:
        if asyncio.get_event_loop().time() - start >= timeout:
            raise TimeoutError("Frontend connection timed out")

        await asyncio.sleep(0.25)

async def open_update_websockets(ip, port, name):
    global ERROR_MESSAGE
    RUST_WS_RETRIES = 3
    uri = f"ws://{ip}:{port}"
    last_error = None

    for attempt in range(1, RUST_WS_RETRIES + 1):
        try:
            await broadcast(status(f"Attempting to connect to ({name}) {attempt}/{RUST_WS_RETRIES}"))
            await asyncio.sleep(2)

            ws = await asyncio.wait_for(
                websockets.connect(uri),
                timeout=5
            )

            pong = ws.ping()
            await asyncio.wait_for(pong, timeout=2)

            await broadcast(status(f"({name}) connected successfully"))
            return ws

        except Exception as e:
            last_error = e
            await broadcast(status(f"({name}) connection attempt {attempt} failed"))

            if attempt < RUST_WS_RETRIES:
                await asyncio.sleep(2)

    await broadcast(status(f"({name}) failed to connect, force exit! "))
    await broadcast(error(True))
    await asyncio.sleep(2)
    ERROR_MESSAGE += f"Server ({name}):\n\n"
    ERROR_MESSAGE += "Critical Error:\n"
    ERROR_MESSAGE += str(last_error)
    await broadcast({"type": "done", "success": "false", "message": ERROR_MESSAGE})
    raise RuntimeError(
        f"({name}) failed to connect after {RUST_WS_RETRIES} attempts: {last_error}"
    )

async def connect_all_rust_servers():
    PROGRESS_PERCENTAGE = 20
    rust_connections = {}

    for key, srv in config.servers.items():
        port = UPDATE_PORTS.get(key)
        ip = srv["ip"]
        name = srv["name"]

        await broadcast(progress("panel", PROGRESS_PERCENTAGE))
        PROGRESS_PERCENTAGE = min(PROGRESS_PERCENTAGE + 20, 100)

        ws = await open_update_websockets(ip, port, name)
        rust_connections[key] = ws
        await asyncio.sleep(2)

    await broadcast(progress("panel", 100))
    print(rust_connections)
    return rust_connections

#--- Main Loop ---
async def main():
    async with websockets.serve(ws_handler, "0.0.0.0", PORT):
        print("If you are trying to run this app manually from the file system stop now!\n"
              "Log in to your control panel and start the update process by clicking ⚙ gear icon in the Super Admin Panel.\n\n"
              "If there is something wrong with your control panel, running this app will not fix it!\n"
              "If you're trying to perform many IRCP updates and it keeps failing but you've done no troubleshooting... YOU'RE INSANE!\n\n"
              f"Websocket running on port: {PORT}")

        try:
            await wait_for_frontend()
            await clear_console()
            await asyncio.sleep(3)
            ws_connections = await connect_all_rust_servers()
            await asyncio.sleep(3)
            await run_rust_server_update(ws_connections)
            await asyncio.sleep(3)
            await run_panel_update()

        except Exception as e:
            print(f"Update aborted: {e}")
            return
        
        finally:
            await asyncio.sleep(10)

asyncio.run(main())
