import subprocess
import platform
import sys
import time
import os
import ipaddress
import pprint
import psutil
import json

BASE_DIR = os.path.dirname(os.path.abspath(__file__))
PANEL_USERS_FILE = os.path.join(BASE_DIR, "panelusers.json")
PY_CONFIG_FILE = os.path.join(BASE_DIR, "config.py")
JS_CONFIG_FILE = os.path.join(BASE_DIR, "config.js")

GROUP_NAME_MAX_LENGTH = 18
SERVER_NAME_MAX_LENGTH = 14

SERVER_INDEX_TO_NAME = {
    1: "Server_One",
    2: "Server_Two",
    3: "Server_Three",
    4: "Server_Four"
}

NUMBER_INDEX_TO_WORD = {
    1: "One",
    2: "Two",
    3: "Three",
    4: "Four"
}

servers = {}

def clear_console():
    os.system('cls' if os.name == 'nt' else 'clear')

def print_banner():
    print(r"""
================================================================================
================================================================================
====                                                                        ====
====    ██╗██████╗  ██████╗██████╗                                          ====
====    ██║██╔══██╗██╔════╝██╔══██╗                                         ====
====    ██║██████╔╝██║     ██████╔╝   Integrated Rust Control Panel         ====
====    ██║██╔══██╗██║     ██╔═══╝    Brought to you by Im_just_a_Pixel     ====
====    ██║██║  ██║╚██████╗██║                                              ====
====    ╚═╝╚═╝  ╚═╝ ╚═════╝╚═╝                                              ====
====                                                                        ====
================================================================================
================================================================================
""")

def init_text():
    print("Shortly you will be prompted to answer several questions.")
    print("When the > arrow appears, simply start typing your answer.")
    print("Once you have finished typing your answer press the ENTER key.")
    print("Alright, lets get started.")
    print()
    print("--------------------------------------------------------------------------------")

def setup_config_text():
    print("In order to get your control panel connected to your Rust servers, we first need to input some networking details.")
    print()
    time.sleep(8)
    print("While we're at it, lets also give your Rust servers some nicknames so you can identify them on the control panel.")
    print()
    time.sleep(8)

def setup_admin_text():
    print("We'll set up your owner admin account which you will use to login and access the control panel.")
    print()
    time.sleep(8)

def start_config_setup():
    print("What is your group or clan name? This will be used to give your control panel a name.")
    while True:
        group_name = input("> ").strip()

        if not group_name:
            print("Name cannot be empty. Please try again.")
            continue

        if len(group_name) > GROUP_NAME_MAX_LENGTH:
            print(f"Name cannot be longer than {GROUP_NAME_MAX_LENGTH} characters. Please try again.")
            continue

        if not group_name.isascii():
            print("Name may only contain ASCII characters. Please remove special or accented characters.")
            continue

        break

    clear_console()
    print_banner()
    init_text()

    for server_index in range(1, 4 + 1):
        time.sleep(1)
        print(f"Give your Rust server a nickname. This will be used on the control panel to identify server #{server_index}.")
        while True:
            server_name = input("> ").strip()

            if not server_name:
                print("Name cannot be empty. Please try again.")
                continue

            if len(server_name) > SERVER_NAME_MAX_LENGTH:
                print(f"Name cannot be longer than {SERVER_NAME_MAX_LENGTH} characters. Please try again.")
                continue

            if not server_name.isascii():
                print("Name can only contain ASCII characters. Please remove special or accented characters.")
                continue

            break

        clear_console()
        print_banner()
        init_text()

        time.sleep(1)
        print(f"Please enter the LAN or WAN IPv4 address for Rust server #{server_index}.")
        time.sleep(1)
        print("Use the Rust Server's LAN IP if it is on the same network as the control panel.")
        time.sleep(1)
        print("Use the WAN IP if the Rust server is located somewhere else on the internet.")

        while True:
            server_ip = input("> ").strip()

            if not server_ip:
                print("IP cannot be empty. Please try again.")
                continue

            try:
                ipaddress.IPv4Address(server_ip)
            except ipaddress.AddressValueError:
                print("Invalid IPv4 address. Please enter a valid IPv4 address (e.g. 192.168.1.100).")
                continue

            break

        server_key = SERVER_INDEX_TO_NAME[server_index]
        servers[server_key] = {
            "name": server_name,
            "ip": server_ip
        }

        if server_index == 4:
            break

        clear_console()
        print_banner()
        init_text()

        time.sleep(1)
        print("Would you like to configure another Rust server? Type yes or no.")

        while True:
            setup_next_server = input("> ").strip().lower()

            if setup_next_server in ("yes", "y"):
                clear_console()
                print_banner()
                init_text()
                break

            if setup_next_server in ("no", "n"):
                break

            print("Please type 'yes' or 'no'.")

        if setup_next_server in ("no", "n"):
            break
        
    while True:
        review_configuration(group_name, servers)

        print("Is this information correct? Type yes or no.")
        confirm = input("> ").strip().lower()

        if confirm in ("yes", "y"):
            clear_console()
            print_banner()
            print("\nConfiguration confirmed.")
            time.sleep(3)
            restart_servers = kill_backends()
            write_config_file(servers)
            write_js_config_file(group_name, servers)
            print("Config files have been created successfully.")
            time.sleep(3)

            if restart_servers:
                start_script("BackendListener.py")
                print("Restarting BackendListener")
                time.sleep(1)
                start_script("BackendLogs.py")
                print("Restarting BackendLogs")
                time.sleep(1)
            
            break

        if confirm not in ("no", "n"):
            print("Please type 'yes' or 'no'.")
            continue

        clear_console()
        print_banner()
        show_amend_menu(servers)

        while True:
            choice = input("> ").strip()

            if not choice.isdigit():
                print("Please enter a number from the list.")
                continue

            choice = int(choice)
            back_option_number = len(servers) + 2

            if choice == back_option_number:
                break

            if choice == 1:
                clear_console()
                print_banner()
                print("Enter new Control Panel Name:")
                group_name = prompt_group_name()
                break

            server_keys = list(servers.keys())
            server_choice_index = choice - 2
            if 0 <= server_choice_index < len(server_keys):
                clear_console()
                print_banner()

                server_key = server_keys[server_choice_index]
                server = servers[server_key]
                new_name, new_ip = prompt_server_details(server_choice_index + 1)

                server["name"] = new_name
                server["ip"] = new_ip
                break

            print("Invalid selection. Please choose a valid number.")

def start_admin_setup():
    while True:
        print("What will be your control panel account username? This should match your Rust In-Game-Name")
        print("as some of the control panel's features will depend on this.")
        while True:
            username = input("> ").strip()
            if not username:
                print("Username cannot be empty.")
                continue
            if not username.isascii():
                print("Username must be ASCII only.")
                continue
            break

        clear_console()
        print_banner()
        init_text()
        time.sleep(1)
        print("Please provide a password for the account. This should be unique. DO NOT use any existing")
        print("password you have for Steam, social media, other games, tik tok, or anything else!")
        while True:
            password = input("> ").strip()
            if not password:
                print("Password cannot be empty.")
                continue
            break

        while True:
            review_admin_account(username, password)

            print("Is this information correct? Type yes or no.")
            confirm = input("> ").strip().lower()

            if confirm in ("yes", "y"):
                create_panel_users(username, password)
                clear_console()
                print_banner()
                print("Owner account created successfully.")
                time.sleep(2)
                return
            

            if confirm in ("no", "n"):
                clear_console()
                print_banner()
                review_admin_account(username, password)
                print("Let's reset and try again.")
                time.sleep(5)
                clear_console()
                print_banner()
                init_text()
                break

def review_configuration(group_name, servers):
    clear_console()
    print_banner()

    print("CONFIGURATION REVIEW")
    print("--------------------------------------------------------------------------------")
    print(f"Control Panel Name:")
    print(f"  {group_name}")
    print()

    for idx, (server_key, server) in enumerate(servers.items(), start=1):
        print(f"Rust Server #{idx}")
        print(f"  Nickname : {server['name']}")
        print(f"  IP       : {server['ip']}")
        print()

    print("--------------------------------------------------------------------------------")

def review_admin_account(username, password):
    clear_console()
    print_banner()

    print("OWNER ACCESS REVIEW")
    print("--------------------------------------------------------------------------------")
    print("Owner / Admin Account:")
    print(f"  Username : {username}")
    print(f"  Password : {password}")
    print()
    print("--------------------------------------------------------------------------------")


def show_amend_menu(servers):
    print("Which item needs to be amended?")
    print("--------------------------------------------------------------------------------")
    print("1. Control Panel Name")

    for idx, (server_key, server) in enumerate(servers.items(), start=2):
        print(f"{idx}. Rust Server #{idx - 1} ({server['name']})")

    print(f"{len(servers) + 2}. Back")
    print("--------------------------------------------------------------------------------")

def prompt_group_name():    
    while True:
        group_name = input("> ").strip()

        if not group_name:
            print("Name cannot be empty. Please try again.")
            continue

        if len(group_name) > GROUP_NAME_MAX_LENGTH:
            print(f"Name cannot be longer than {GROUP_NAME_MAX_LENGTH} characters. Please try again.")
            continue

        if not group_name.isascii():
            print("Name may only contain ASCII characters.")
            continue

        return group_name

def prompt_server_details(server_number):
    print(f"Enter new Rust server #{server_number} details:")

    while True:
        server_name = input("Server nickname > ").strip()

        if not server_name:
            print("Name cannot be empty.")
            continue

        if len(server_name) > SERVER_NAME_MAX_LENGTH:
            print(f"Name cannot exceed {SERVER_NAME_MAX_LENGTH} characters.")
            continue

        if not server_name.isascii():
            print("Name may only contain ASCII characters.")
            continue

        break

    while True:
        server_ip = input("Server IPv4 address > ").strip()

        if not server_ip:
            print("IP cannot be empty.")
            continue

        try:
            ipaddress.IPv4Address(server_ip)
        except ipaddress.AddressValueError:
            print("Invalid IPv4 address.")
            continue

        break

    return server_name, server_ip

def write_config_file(servers):
    config_path = os.path.join(os.path.dirname(__file__), "config.py")

    with open(config_path, "w", encoding="utf-8") as f:
        f.write("# Auto-generated Rust Control Panel configuration\n\n")
        f.write("servers = ")
        f.write(pprint.pformat(servers, indent=4))
        f.write("\n")

def write_js_config_file(group_name, servers):
    js_path = os.path.join(os.path.dirname(__file__), "config.js")

    with open(js_path, "w", encoding="utf-8") as f:
        f.write("// Auto-generated Rust Control Panel configuration\n\n")
        f.write(f'var controlPanelName = "{group_name}";\n')

        for index, (server_key, server) in enumerate(servers.items(), start=1):
            word = NUMBER_INDEX_TO_WORD[index]
            js_prefix = f"server{word}"

            f.write(f'var {js_prefix}Name = "{server["name"]}";\n')

def create_panel_users(username, password):
    users_path = os.path.join(os.path.dirname(__file__), "panelusers.json")

    default_permissions = {
        "giveTo": True,
        "giveAll": True,
        "teleportPlayer": True,
        "teleportYou": True,
        "killPlayer": True,
        "respawnPlayer": True,
        "gameTime": True,
        "weatherCloudy": True,
        "weatherRainy": True,
        "weatherFoggy": True,
        "weatherReset": True,
        "mutePlayer": True,
        "unmutePlayer": True,
        "kickPlayer": True,
        "kickAll": True,
        "banPlayer": True,
        "unbanPlayer": True,
        "fastRestart": True,
        "delayedRestart": True,
        "serverStart": True,
        "serverStop": True,
        "makeAdmin": True,
        "removeAdmin": True,
        "fullWipe": True,
        "halfWipe": True,
        "modManager": True,
        "serverSettings": True,
        "fileSystem": True,
        "clearServerLogs": True,
        "clearPanelLogs": True,
        "addCpUser": True,
        "removeCpUser": True,
        "modifyCpUser": True,
        "customCommand": True,
        "eventSpawning": True,
        "mngPlayerHistory": True,
        "miniMap": True,
        "minimapCustomTp": True,
        "updateCtrlPanel": True,
        "superAdmin": True,
        "owner": True
    }
    
    users = []

    users.append({
        "username": username,
        "password": password,
        "permissions": default_permissions
    })    

    with open(users_path, 'w') as f:
        json.dump(users, f, indent=2)        

def start_script(script_name):
    if platform.system() == "Windows":
        si = subprocess.STARTUPINFO()
        si.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        si.wShowWindow = 2

        subprocess.Popen(
            [sys.executable, script_name],
            startupinfo=si,
            creationflags=subprocess.CREATE_NEW_CONSOLE
        )
    else:
        subprocess.Popen([sys.executable, script_name])

def kill_backends():
    TARGET_SCRIPTS = {
        "BackendListener.py",
        "BackendLogs.py"
    }

    killed = []

    for proc in psutil.process_iter(['pid', 'name', 'cmdline']):
        try:
            cmdline = proc.info['cmdline']
            if not cmdline:
                continue

            for arg in cmdline:
                base = os.path.basename(arg)
                if base in TARGET_SCRIPTS:
                    print(f"Killing {base} (PID {proc.pid})")
                    proc.terminate()
                    proc.wait(timeout=5)
                    killed.append(base)
                    break

        except (psutil.NoSuchProcess, psutil.AccessDenied):
            continue

    if killed:
        print("Terminated stale process:", ", ".join(sorted(set(killed))))
        time.sleep(2)
        return True

    return False

def main():
    did_we_config = False
    print_banner()
    
    time.sleep(5)
    print("Welcome to the Rust Server Control Panel configurator.")
    print()
    time.sleep(5)

    if os.path.exists(PY_CONFIG_FILE) and os.path.exists(JS_CONFIG_FILE) and os.path.exists(PANEL_USERS_FILE):
        setup_config_text()

    if not os.path.exists(PY_CONFIG_FILE) or not os.path.exists(JS_CONFIG_FILE):
        setup_config_text()

    if not os.path.exists(PANEL_USERS_FILE):
        setup_admin_text()

    clear_console()
    print_banner()

    time.sleep(3)
    print("Shortly you will be prompted to answer some questions.")
    time.sleep(3)
    print("When the > arrow appears, simply start typing your answer.")
    time.sleep(3)
    print("Once you have finished typing your answer press the ENTER key.")

    time.sleep(3)
    print("Alright, lets get started.")
    print()
    time.sleep(3)
    print("--------------------------------------------------------------------------------")
    time.sleep(1)

    if os.path.exists(PY_CONFIG_FILE) and os.path.exists(JS_CONFIG_FILE) and os.path.exists(PANEL_USERS_FILE):
        start_config_setup()

    if not os.path.exists(PY_CONFIG_FILE) or not os.path.exists(JS_CONFIG_FILE):
        did_we_config = True
        start_config_setup()
        clear_console()
        print_banner()

    if not os.path.exists(PANEL_USERS_FILE):
        if did_we_config:
            init_text()

        start_admin_setup()

    print("Configurator will now exit...")
    time.sleep(5)
                
if __name__ == "__main__":
    main()

