from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler
from datetime import datetime
from pathlib import Path
import asyncio
import websockets
import os
import json
import re
import math
import config
import ctypes
import time
import psutil

ctypes.windll.kernel32.SetConsoleTitleW("BackendLogs")

# Network Port
SERVER_PORT = 28002

# Rust Log server address config
SERVER_ONE_IP = config.servers.get("Server_One", {}).get("ip")
SERVER_TWO_IP = config.servers.get("Server_Two", {}).get("ip")
SERVER_THREE_IP = config.servers.get("Server_Three", {}).get("ip")
SERVER_FOUR_IP = config.servers.get("Server_Four", {}).get("ip")

# File Directories
BASE_DIR = Path(__file__).resolve().parent
PANEL_LOGS = BASE_DIR / "panellogs.log"
RUST_SERVERONE_LOG = BASE_DIR / "rustserverone.log"
RUST_SERVERTWO_LOG = BASE_DIR / "rustservertwo.log"
RUST_SERVERTHREE_LOG = BASE_DIR / "rustserverthree.log"
RUST_SERVERFOUR_LOG = BASE_DIR / "rustserverfour.log"
MESSAGES = BASE_DIR / "messages.txt"

# === Config ===
LOG_FILES = {
    "/panel": PANEL_LOGS,
    "/Logs_Server_One": RUST_SERVERONE_LOG,
    "/Logs_Server_Two": RUST_SERVERTWO_LOG,
    "/Logs_Server_Three": RUST_SERVERTHREE_LOG,
    "/Logs_Server_Four": RUST_SERVERFOUR_LOG,
    "/Players_Server_One": None,
    "/Players_Server_Two": None,
    "/Players_Server_Three": None,
    "/Players_Server_Four": None
}

# In-memory player stores
online_players_serverone = {}
online_players_servertwo = {}
online_players_serverthree = {}
online_players_serverfour = {}
players_clients_serverone = set()
players_clients_servertwo = set()
players_clients_serverthree = set()
players_clients_serverfour = set()

HISTORY_FILES = {
    "Server_One": BASE_DIR / "historyserverone.json",
    "Server_Two": BASE_DIR / "historyservertwo.json",
    "Server_Three": BASE_DIR / "historyserverthree.json",
    "Server_Four": BASE_DIR / "historyserverfour.json"
}

# Message Center
message_clients = set()
MESSAGES_FILE = MESSAGES

# Log Stream Server addresses
EXTERNAL_SERVERS = [
    {
        "uri": f"ws://{SERVER_ONE_IP}:28011/",
        "log_path": RUST_SERVERONE_LOG,
        "server": "Server_One"
    },
    {
        "uri": f"ws://{SERVER_TWO_IP}:28021/",
        "log_path": RUST_SERVERTWO_LOG,
        "server": "Server_Two"
    },
    {
        "uri": f"ws://{SERVER_THREE_IP}:28031/",
        "log_path": RUST_SERVERTHREE_LOG,
        "server": "Server_Three"
    },
    {
        "uri": f"ws://{SERVER_FOUR_IP}:28041/",
        "log_path": RUST_SERVERFOUR_LOG,
        "server": "Server_Four"
    }
]

# Minimap Servers
RELAY_TARGETS = {
    "Server_One": f"ws://{SERVER_ONE_IP}:28012/map",
    "Server_Two": f"ws://{SERVER_TWO_IP}:28022/map",
    "Server_Three": f"ws://{SERVER_THREE_IP}:28032/map",
    "Server_Four": f"ws://{SERVER_FOUR_IP}:28042/map"
}

# Info Services
INFO_UPSTREAMS = {
    "Server_One": [
        f"ws://{SERVER_ONE_IP}:28011/info",  # slot a
        f"ws://{SERVER_ONE_IP}:28013"        # slot b
    ],
    "Server_Two": [
        f"ws://{SERVER_TWO_IP}:28021/info",  # slot a
        f"ws://{SERVER_TWO_IP}:28023"        # slot b
    ],
    "Server_Three": [
        f"ws://{SERVER_THREE_IP}:28031/info",  # slot a
        f"ws://{SERVER_THREE_IP}:28033"        # slot b
    ],
    "Server_Four": [
        f"ws://{SERVER_FOUR_IP}:28041/info",  # slot a
        f"ws://{SERVER_FOUR_IP}:28043"        # slot b
    ]
}

combined_state = {
    "Server_One": {"a": {}, "b": {}},
    "Server_Two": {"a": {}, "b": {}},
    "Server_Three": {"a": {}, "b": {}},
    "Server_Four": {"a": {}, "b": {}}
}
info_clients = {
    "Server_One": set(),
    "Server_Two": set(),
    "Server_Three": set(),
    "Server_Four": set()
}

# === Server Info Data ===
async def broadcast_combined(server_key):
    while True:
        if server_key in info_clients and info_clients[server_key]:
            merged = {**combined_state[server_key]["a"], **combined_state[server_key]["b"]}
            dead = []
            for client in info_clients[server_key]:
                try:
                    await client.send(json.dumps(merged))
                except:
                    dead.append(client)
            for d in dead:
                info_clients[server_key].discard(d)
        await asyncio.sleep(1)

async def handle_info_client(websocket, server_key):
    print(f"📊 Client connected to /Info_{server_key}")
    info_clients[server_key].add(websocket)
    try:
        merged = {**combined_state[server_key]["a"], **combined_state[server_key]["b"]}
        await websocket.send(json.dumps(merged))
        async for _ in websocket:  # keep alive
            pass
    except Exception as e:
        print(f"⚠️ Error in /Info_{server_key} client: {e}")
    finally:
        info_clients[server_key].discard(websocket)
        print(f"❌ Client disconnected from /Info_{server_key}")

async def handle_info_upstream(uri, server_key, slot):
    PLAYER_STORES = {
        "Server_One": online_players_serverone,
        "Server_Two": online_players_servertwo,
        "Server_Three": online_players_serverthree,
        "Server_Four": online_players_serverfour
    }

    PATH_MAP = {
        "Server_One": "/Players_Server_One",
        "Server_Two": "/Players_Server_Two",
        "Server_Three": "/Players_Server_Three",
        "Server_Four": "/Players_Server_Four"
    }

    player_store = PLAYER_STORES.get(server_key)
    path = PATH_MAP.get(server_key)
    last_successful_connection = datetime.now() if slot != "b" else None

    STALE_TIMEOUT = 30  # seconds
    AUTO_DISCONNECT_THRESHOLD = 300  # Seconds

    while True:
        try:
            async with websockets.connect(uri) as ws:
                print(f"Connected to upstream {slot} for {server_key}: {uri}")

                async for msg in ws:
                    try:
                        data = json.loads(msg)
                        combined_state[server_key][slot] = data

                        if slot == "b" and "players" in data:
                            now = datetime.now()
                            new_players = {str(p["steamid"]): p for p in data["players"]}
                            last_successful_connection = now

                            for steam_id, pdata in new_players.items():
                                if steam_id not in player_store:
                                    player_store[steam_id] = {
                                        "name": pdata["name"],
                                        "steam_id": steam_id,
                                        "ip": pdata["ip"],
                                        "admin": False,
                                        "connected_at": now,
                                        "last_update": now
                                    }
                                    print(f"Player connected: {pdata['name']}")
                                else:
                                    player_store[steam_id]["name"] = pdata["name"]
                                    player_store[steam_id]["ip"] = pdata["ip"]
                                    player_store[steam_id]["last_update"] = now

                            current_ids = set(player_store.keys())
                            new_ids = set(new_players.keys())
                            disconnected_ids = current_ids - new_ids
                            
                            for steam_id in disconnected_ids:
                                player = player_store.pop(steam_id, None)
                                if player:
                                    try:
                                        connected_at = player.get("connected_at")
                                        if connected_at:
                                            session_seconds = int((now - connected_at).total_seconds())
                                        else:
                                            session_seconds = 0  # fallback
                                        update_player_history(player, session_seconds, server_key)
                                        print(f"🔴 Player disconnected: {player['name']} (session {session_seconds}s)")
                                    except Exception as e:
                                        print(f"❌ Failed to update history for {player.get('name','?')}: {e}")

                            stale_ids = []
                            for steam_id, player in player_store.items():
                                last_update = player.get("last_update", now)
                                if (now - last_update).total_seconds() > STALE_TIMEOUT:
                                    stale_ids.append(steam_id)

                            for steam_id in stale_ids:
                                player = player_store.pop(steam_id, None)
                                if player:
                                    session_seconds = int((now - player.get("connected_at", now)).total_seconds())
                                    update_player_history(player, session_seconds, server_key)
                                    print(f"Player disconnected (stale >30s): {player['name']} (session {session_seconds}s)")

                            await broadcast_player_data(path)

                    except Exception as e:
                        print(f"❌ JSON error from {uri}: {e}")

        except Exception as e:
            print(f"⚠️ Upstream {slot} for {server_key} failed: {e}, retrying in 5s...")
            
            if slot == "b" and last_successful_connection:
                down_duration = (datetime.now() - last_successful_connection).total_seconds()
                if down_duration > AUTO_DISCONNECT_THRESHOLD and player_store:
                    print(f"⚠️ Slot B for {server_key} down for {int(down_duration)}s. Disconnecting all players.")
                    now = datetime.now()
                    for steam_id, player in list(player_store.items()):
                        session_seconds = int((now - player.get("connected_at", now)).total_seconds())
                        update_player_history(player, session_seconds, server_key)
                    player_store.clear()
                    
            await asyncio.sleep(5)

# === Broadcast Players ===
async def broadcast_player_data(path):
    if path == "/Players_Server_One":
        clients = players_clients_serverone
        data = list(online_players_serverone.values())
    elif path == "/Players_Server_Two":
        clients = players_clients_servertwo
        data = list(online_players_servertwo.values())
    elif path == "/Players_Server_Three":
        clients = players_clients_serverthree
        data = list(online_players_serverthree.values())
    elif path == "/Players_Server_Four":
        clients = players_clients_serverfour
        data = list(online_players_serverfour.values())
    else:
        return

    if not clients:
        return

    for player in data:
        try:
            delta = datetime.now() - player["connected_at"]
            player["time"] = str(delta).split('.')[0]
        except Exception as e:
            player["time"] = "00:00:00"
            print(f"Time calc error for {player.get('name', '?')}: {e}")

    json_data = json.dumps(data, default=str)
    to_remove = set()

    for ws in clients:
        try:
            await ws.send(json_data)
        except Exception as e:
            print(f"WebSocket send error: {e}")
            to_remove.add(ws)

    clients.difference_update(to_remove)

# === Player Record Keeping ===
def update_player_history(player, session_seconds, server_key):
    history_path = HISTORY_FILES.get(server_key)

    if history_path is None:
        print(f"❌ Invalid server_key: {server_key}")
        return

    steam_id = player["steam_id"]
    now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

    try:
        if history_path.exists():
            with open(history_path, "r", encoding="utf-8") as f:
                history = json.load(f)
                if not isinstance(history, list):
                    print(f"⚠️ History file malformed, resetting to empty list: {history_path}")
                    history = []
        else:
            history = []
    except Exception as e:
        print(f"⚠️ Error reading history file: {e}")
        history = []

    existing = next((p for p in history if p["steam_id"] == steam_id), None)
    if existing:
        existing["name"] = player["name"]
        existing["time_seconds"] += session_seconds
        existing["time"] = format_seconds(existing["time_seconds"])
        existing["last_active"] = now
        existing["ip"] = player["ip"]
        existing["hidden"] = False
    else:
        history.append({
            "name": player["name"],
            "steam_id": steam_id,
            "ip": player["ip"],
            "time_seconds": session_seconds,
            "time": format_seconds(session_seconds),
            "last_active": now,
            "hidden": False
        })

    try:
        with open(history_path, "w", encoding="utf-8") as f:
            json.dump(history, f, indent=2)
    except Exception as e:
        print(f"❌ Error saving history file: {e}")

def format_seconds(seconds):
    hours = seconds // 3600
    minutes = (seconds % 3600) // 60
    secs = seconds % 60
    return f"{int(hours):02}:{int(minutes):02}:{int(secs):02}"

#=== Minimap Socket Relay ===
async def minimap_traffic(websocket):
    try:
        init_msg = await asyncio.wait_for(websocket.recv(), timeout=5)
        init_data = json.loads(init_msg)
        target = init_data.get("target")

        if target not in RELAY_TARGETS:
            await websocket.send(json.dumps({"error": "Invalid relay target."}))
            await websocket.close()
            return

        target_uri = RELAY_TARGETS[target]
        print(f"🔄 Relaying client to {target} ({target_uri})")

        async with websockets.connect(target_uri, max_size=20 * 1024 * 1024) as rust_socket:

            async def from_client():
                try:
                    async for msg in websocket:
                        await rust_socket.send(msg)
                except:
                    pass  # Let outer finally close

            async def from_server():
                try:
                    async for msg in rust_socket:
                        await websocket.send(msg)
                except Exception as e:
                    print(f"[Relay Error] from_server failed: {e}")
                    await websocket.send(json.dumps({"error": f"Relay error: {str(e)}"}))

            await asyncio.gather(from_client(), from_server())

    except asyncio.TimeoutError:
        await websocket.send(json.dumps({"error": "Target not specified"}))
    except Exception as e:
        await websocket.send(json.dumps({"error": f"Relay error: {str(e)}"}))
    finally:
        await websocket.close()

# === Message Centre Socket ===
async def message_handler(websocket):
    print("💬 Client connected to /messages")
    message_clients.add(websocket)

    try:
        await send_existing_messages(websocket)

        async for raw in websocket:
            try:
                msg_data = json.loads(raw)
                msg_data["timestamp"] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

                with open(MESSAGES_FILE, 'a', encoding='utf-8') as f:
                    f.write(json.dumps(msg_data) + '\n')

                dead = set()

                for client in message_clients:
                    try:
                        await client.send(json.dumps(msg_data))
                    except Exception:
                        dead.add(client)

                message_clients.difference_update(dead)

            except Exception as e:
                print(f"❌ Error processing message: {e}")

    except Exception as e:
        print(f"⚠️ WebSocket error (/messages): {e}")
    finally:
        message_clients.discard(websocket)
        print("❌ /messages client disconnected")

async def send_existing_messages(websocket):
    if os.path.exists(MESSAGES_FILE):
        try:
            with open(MESSAGES_FILE, 'r', encoding='utf-8') as f:
                for line in f:
                    await websocket.send(line.strip())
        except Exception as e:
            print(f"⚠️ Failed to send existing messages: {e}")


# === Log Streams ===
async def log_stream(websocket, file_path, path):
    if path == "/Players_Server_One":
        players_clients_serverone.add(websocket)
        print("🔌 Client connected to Players_Server_One")
        try:
            while True:
                await broadcast_player_data("/Players_Server_One")
                await asyncio.sleep(1)
        finally:
            players_clients_serverone.discard(websocket)
            print("❌ Client disconnected from Players_Server_One")

    elif path == "/Players_Server_Two":
        players_clients_servertwo.add(websocket)
        print("🔌 Client connected to Players_Server_Two")
        try:
            while True:
                await broadcast_player_data("/Players_Server_Two")
                await asyncio.sleep(1)
        finally:
            players_clients_servertwo.discard(websocket)
            print("❌ Client disconnected from Players_Server_Two")

    elif path == "/Players_Server_Three":
        players_clients_serverthree.add(websocket)
        print("🔌 Client connected to Players_Server_Three")
        try:
            while True:
                await broadcast_player_data("/Players_Server_Three")
                await asyncio.sleep(1)
        finally:
            players_clients_serverthree.discard(websocket)
            print("❌ Client disconnected from Players_Server_Three")

    elif path == "/Players_Server_Four":
        players_clients_serverfour.add(websocket)
        print("🔌 Client connected to Players_Server_Four")
        try:
            while True:
                await broadcast_player_data("/Players_Server_Four")
                await asyncio.sleep(1)
        finally:
            players_clients_serverfour.discard(websocket)
            print("❌ Client disconnected from Players_Server_Four")

    else:
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                for line in f:
                    await websocket.send(line)
                last_size = f.tell()

                while True:
                    current_size = os.path.getsize(file_path)
                    if current_size < last_size:
                        f.close()
                        await asyncio.sleep(0.25)
                        f = open(file_path, 'r', encoding='utf-8')
                        f.seek(0)
                        line = f.readlines()
                        await websocket.send(line[-1])
                        last_size = f.tell()

                    line = f.readline()
                    if line:
                        await websocket.send(line)
                    last_size = f.tell()
                    await asyncio.sleep(0.05)
        except Exception as e:
            await websocket.send(f"⚠️ Log stream error: {str(e)}")

# === WebSocket Handler ===
async def handler(websocket):
    path = websocket.request.path

    if path == "/minimap":
        await minimap_traffic(websocket)
    elif path == "/messages":
        await message_handler(websocket)
    elif path == "/Info_Server_One":
        await handle_info_client(websocket, "Server_One")
    elif path == "/Info_Server_Two":
        await handle_info_client(websocket, "Server_Two")
    elif path == "/Info_Server_Three":
        await handle_info_client(websocket, "Server_Three")
    elif path == "/Info_Server_Four":
        await handle_info_client(websocket, "Server_Four")
    elif path in ["/Players_Server_One", "/Players_Server_Two", "/Players_Server_Three", "/Players_Server_Four"]:
        await log_stream(websocket, None, path)
    elif path in LOG_FILES and LOG_FILES[path] and os.path.isfile(LOG_FILES[path]):
        await log_stream(websocket, LOG_FILES[path], path)
    else:
        await websocket.send(json.dumps({"error": "Invalid path or log file not found."}))
        await websocket.close()

# === Redundant logging code but keeping for a while for troubleshooting purpose
"""
async def receive_and_log(uri, log_path, server_key):
    PLAYER_STORES = {
        "Server_One": online_players_serverone,
        "Server_Two": online_players_servertwo,
        "Server_Three": online_players_serverthree,
        "Server_Four": online_players_serverfour
    }

    os.makedirs(os.path.dirname(log_path), exist_ok=True)

    join_pattern_1 = re.compile(r"^(?P<name>.+?)\s+with\s+steamid\s+(?P<steam_id>\d+)\s+joined\s+from\s+ip\s+(?P<ip>[\d\.]+):\d+")
    join_pattern_2 = re.compile(r"^(?P<ip>[\d\.]+):\d+/(?P<steam_id>\d+)/(?P<name>\S+)\s+joined")
    disconnect_events = [
        {
            # Example: 203.59.253.229:55364/76561198000000000/Name disconnecting:
            "pattern": re.compile(r"^\d+\.\d+\.\d+\.\d+:\d+/(?P<steam_id>\d{17})/(?P<name>.+?)\s+disconnecting:"),
            "extract": lambda m: m.group("steam_id")
        },
        {
            # Example: Kicked: Nathan Explosion
            "pattern": re.compile(r"^Kicked:\s+(?P<name>.+)$"),
            "extract": lambda m: next(
                (sid for sid, p in player_store.items() if p["name"] == m.group("name")), None
            )
        },
        {
            # Example: Kickbanned User: 76561198000000000 - Nathan Explosion
            "pattern": re.compile(r"^Kickbanned User:\s+(?P<steam_id>\d{17})\s+-\s+(?P<name>.+)$"),
            "extract": lambda m: m.group("steam_id")
        },
        {
            # Example: SomeName[76561198000000000] FlyHack: ...
            "pattern": re.compile(r"^\S+\[(?P<steam_id>\d{17})\]\s+FlyHack:"),
            "extract": lambda m: m.group("steam_id")
        },
        {
            # Example: Kicking 203.59.253.229:55364/76561198000000000/Name With Space
            "pattern": re.compile(r"^Kicking\s+\d+\.\d+\.\d+\.\d+:\d+/(?P<steam_id>\d{17})/(?P<name>.+)"),
            "extract": lambda m: m.group("steam_id")
        }
    ]

    player_store = PLAYER_STORES.get(server_key)

    status_header_pattern = re.compile(r"id\s+name\s+ping\s+connected\s+addr\s+owner\s+violation\s+kicks", re.IGNORECASE)
    status_data_pattern = re.compile(
        r'^(?P<steam_id>\d{17})\s+"(?P<name>[^"]+)"\s+\d+\s+[\d\.]+s\s+(?P<ip>[\d\.]+):\d+'
    )
    status_dump_active = False
    new_status_players = {}

    while True:
        try:
            async with websockets.connect(uri) as websocket:
                print(f"✅ Connected to logs: {uri}")

                with open(log_path, 'a', encoding='utf-8') as log_file:
                    while True:
                        try:
                            msg = await websocket.recv()
                            if status_header_pattern.search(msg) or status_data_pattern.match(msg.strip()):
                                pass
                            else:
                                log_entry = f"{datetime.now().strftime('[%Y-%m-%d %H:%M:%S]')} {msg.strip()}\n"
                                log_file.write(log_entry)
                                log_file.flush()

                            # === Status Header Line ===
                            if status_header_pattern.search(msg):
                                status_dump_active = True
                                new_status_players.clear()
                                continue

                            # === Status Dump Parsing ===
                            if status_dump_active:
                                m = status_data_pattern.match(msg.strip())
                                if m:
                                    steam_id = m.group("steam_id")
                                    name = m.group("name")
                                    ip = m.group("ip")
                                    existing = player_store.get(steam_id)
                                    new_status_players[steam_id] = True

                                    if not existing:
                                        print(f"🟢 Status log added player: {name}")
                                        player_store[steam_id] = {
                                            "name": name,
                                            "steam_id": steam_id,
                                            "ip": ip,
                                            "admin": False,
                                            "connected_at": datetime.now()
                                        }

                                else:
                                    # End of status dump section
                                    status_dump_active = False

                                    # Disconnect players no longer listed
                                    current_steam_ids = set(player_store.keys())
                                    seen_steam_ids = set(new_status_players.keys())
                                    stale_ids = current_steam_ids - seen_steam_ids

                                    for stale_id in stale_ids:
                                        player = player_store.pop(stale_id, None)
                                        if player and "connected_at" in player:
                                            session_seconds = int((datetime.now() - player["connected_at"]).total_seconds())
                                            update_player_history(player, session_seconds, server_key)
                                            print(f"🔴 Removed stale player: {player['name']}")

                                    continue

                            # === Restart Detected (disconnect all) ===
                            if msg.strip() in ["🔄 Restarting server...", "🖥 Restarting the server  console..."]:
                                print("🔄 Restart message received. Disconnecting all players.")
                                for steam_id, player in list(player_store.items()):
                                    if "connected_at" in player:
                                        session_seconds = int((datetime.now() - player["connected_at"]).total_seconds())
                                        update_player_history(player, session_seconds, server_key)
                                player_store.clear()
                                continue

                            # === Join ===
                            player = None
                            if (m := join_pattern_1.match(msg)) or (m := join_pattern_2.match(msg)):
                                player = {
                                    "name": m.group("name"),
                                    "steam_id": m.group("steam_id"),
                                    "ip": m.group("ip"),
                                    "admin": False,
                                    "connected_at": datetime.now()
                                }

                            if player:
                                player_store[player["steam_id"]] = player
                                continue

                            # === DISCONNECT ===
                            for entry in disconnect_events:
                                match = entry["pattern"].match(msg)
                                if match:
                                    steam_id = entry["extract"](match)
                                    if not steam_id:
                                        continue
                                    player = player_store.pop(steam_id, None)
                                    if player and "connected_at" in player:
                                        session_seconds = int((datetime.now() - player["connected_at"]).total_seconds())
                                        update_player_history(player, session_seconds, server_key)
                                    break

                        except websockets.ConnectionClosed:
                            print(f"⚠️ Disconnected from {uri}")
                            break
                        except Exception as e:
                            print(f"❌ Log recv error: {e}")
                            break

        except Exception as e:
            print(f"❌ Connection failed to {uri}: {e}")
        await asyncio.sleep(5)
"""

# === Log Receiver / Filtering ===
async def receive_and_log(uri, log_path, server_key):
    PLAYER_STORES = {
        "Server_One": online_players_serverone,
        "Server_Two": online_players_servertwo,
        "Server_Three": online_players_serverthree,
        "Server_Four": online_players_serverfour
    }

    player_store = PLAYER_STORES.get(server_key)
    
    os.makedirs(os.path.dirname(log_path), exist_ok=True)

    while True:
        try:
            async with websockets.connect(uri) as websocket:
                print(f"✅ Connected to logs: {uri}")

                with open(log_path, 'a', encoding='utf-8') as log_file:
                    while True:
                        try:
                            msg = await websocket.recv()
                            log_entry = f"{datetime.now().strftime('[%Y-%m-%d %H:%M:%S]')} {msg.strip()}\n"
                            log_file.write(log_entry)
                            log_file.flush()

                            # === Restart Detected (disconnect all) ===
                            if msg.strip() in ["🔄 Restarting server...", "🖥 Restarting the server console..."]:
                                print("🔄 Restart message received. Disconnecting all players.")
                                for steam_id, player in list(player_store.items()):
                                    if "connected_at" in player:
                                        session_seconds = int((datetime.now() - player["connected_at"]).total_seconds())
                                        update_player_history(player, session_seconds, server_key)
                                player_store.clear()
                                continue

                        except websockets.ConnectionClosed:
                            print(f"⚠️ Disconnected from {uri}")
                            break
                        except Exception as e:
                            print(f"❌ Log recv error from {uri}: {e}")
                            break

        except Exception as e:
            print(f"❌ Connection failed to {uri}: {e}, retrying in 5s...")
        await asyncio.sleep(5)

# === Start External Clients ===
async def run_all_external_clients():
    tasks = [
        asyncio.create_task(receive_and_log(cfg["uri"], cfg["log_path"], cfg["server"]))
        for cfg in EXTERNAL_SERVERS
    ]

    for key, uris in INFO_UPSTREAMS.items():
        asyncio.create_task(handle_info_upstream(uris[0], key, "a"))
        asyncio.create_task(handle_info_upstream(uris[1], key, "b"))

    await asyncio.gather(*tasks)

# === Main Runner ===
async def main():
    server = await websockets.serve(handler, "0.0.0.0", SERVER_PORT)
    print(f"✅ WebSocket server started on port {SERVER_PORT}")

    for server_key in combined_state.keys():
        asyncio.create_task(broadcast_combined(server_key))

    await asyncio.gather(
        server.wait_closed(),
        run_all_external_clients()
    )

if __name__ == "__main__":
    asyncio.run(main())  
