from pathlib import Path
import http.server
import http.client
import socketserver
import socket
import requests
import logging
import os
import json
import config
import ctypes
import subprocess
import sys
import time
import platform
import psutil
import cgi
import io

ctypes.windll.kernel32.SetConsoleTitleW("BackendListener")

#Backend Port
PORT = 28001

#Rust Rcon server PORT config
SERVERONE_RCON_PORT = 28010
SERVERTWO_RCON_PORT = 28020
SERVERTHREE_RCON_PORT = 28030
SERVERFOUR_RCON_PORT = 28040

#Server Ops Ports
SERVERONE_OPS_PORT = 28016
SERVERTWO_OPS_PORT = 28026
SERVERTHREE_OPS_PORT = 28036
SERVERFOUR_OPS_PORT = 28046

#Backend file directories
BASE_DIR = Path(__file__).resolve().parent
FILE_CONTROLPANEL_USERS = BASE_DIR / "panelusers.json"
FILE_CONTROLPANEL_LOGS = BASE_DIR / "panellogs.log"
FILE_PLAYERHISTORY_ONE = BASE_DIR / "historyserverone.json"
FILE_PLAYERHISTORY_TWO = BASE_DIR / "historyservertwo.json"
FILE_PLAYERHISTORY_THREE = BASE_DIR / "historyserverthree.json"
FILE_PLAYERHISTORY_FOUR = BASE_DIR / "historyserverfour.json"
FILE_RUSTLOGS_ONE = BASE_DIR / "rustserverone.log"
FILE_RUSTLOGS_TWO = BASE_DIR / "rustservertwo.log"
FILE_RUSTLOGS_THREE = BASE_DIR / "rustserverthree.log"
FILE_RUSTLOGS_FOUR = BASE_DIR / "rustserverfour.log"

FILE_RUSTLOGS = {
    "Server_One": FILE_RUSTLOGS_ONE,
    "Server_Two": FILE_RUSTLOGS_TWO,
    "Server_Three": FILE_RUSTLOGS_THREE,
    "Server_Four": FILE_RUSTLOGS_FOUR
}

#Format URL's
RCON_PORTS = {
    "Server_One": SERVERONE_RCON_PORT,
    "Server_Two": SERVERTWO_RCON_PORT,
    "Server_Three": SERVERTHREE_RCON_PORT,
    "Server_Four": SERVERFOUR_RCON_PORT
}

OPS_PORTS = {
    "Server_One": SERVERONE_OPS_PORT,
    "Server_Two": SERVERTWO_OPS_PORT,
    "Server_Three": SERVERTHREE_OPS_PORT,
    "Server_Four": SERVERFOUR_OPS_PORT
}

SERVER_URLS = {}
for key, srv in config.servers.items():
    port = RCON_PORTS.get(key)
    base = f"http://{srv['ip']}:{port}"
    SERVER_URLS[key] = {
        "name": srv["name"],
        "base": base,
        "fast": f"{base}/Fast_Restart",
        "delayed": f"{base}/Delayed_Restart",
        "custom": f"{base}/Custom_Command",
        "fullwipe": f"{base}/Full_Wipe",
        "halfwipe": f"{base}/Half_Wipe",
        "addmod": f"{base}/Add_Mod",
        "say": f"{base}/Say",
        "update": f"{base}/Update_Rust",
        "rust_log": FILE_RUSTLOGS.get(key)
    }

# Format logging
logging.basicConfig(
    filename=FILE_CONTROLPANEL_LOGS,
    level=logging.INFO,
    format='[%(asctime)s] - %(levelname)s - %(message)s',
    datefmt='%Y-%m-%d %H:%M:%S'
)

FILE_PLAYERHISTORY = {
    "/History_Server_One": FILE_PLAYERHISTORY_ONE,
    "/History_Server_Two": FILE_PLAYERHISTORY_TWO,
    "/History_Server_Three": FILE_PLAYERHISTORY_THREE,
    "/History_Server_Four": FILE_PLAYERHISTORY_FOUR,
}

class BackendRequestHandler(http.server.SimpleHTTPRequestHandler):
    def do_GET(self):
        self.send_response(405)
        self.send_header('Allow', 'POST, PUT, DELETE')
        self.end_headers()
        self.wfile.write(b"GET method is not allowed on this server.")

    def do_OPTIONS(self):
        self.send_response(200)
        self.send_header('Access-Control-Allow-Origin', '*')
        self.send_header('Access-Control-Allow-Methods', 'PUT, POST, OPTIONS')
        self.send_header('Access-Control-Allow-Headers', 'Authorization, Content-Type, X-Target-Server, X-Command, X-User')
        self.end_headers()

    def do_POST(self):
        from urllib.parse import urlparse
        parsed_path = urlparse(self.path)

        if parsed_path.path == '/login':
            content_length = int(self.headers.get('Content-Length', 0))
            post_data = self.rfile.read(content_length).decode('utf-8')

            try:
                payload = json.loads(post_data)
                username = payload.get('username')
                password = payload.get('password')

                if not username or not password:
                    self.send_response(400)
                    self.send_header('Content-Type', 'application/json')
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(json.dumps({
                        "success": False,
                        "message": "Missing username or password."
                    }).encode('utf-8'))
                    return

                with open(FILE_CONTROLPANEL_USERS, 'r') as f:
                    users = json.load(f)
                    for user in users:
                        if user.get('username') == username and user.get('password') == password:
                            permissions = user.get("permissions", {})
                            response = {
                                "success": True,
                                "permissions": permissions
                            }

                            self.send_response(200)
                            self.send_header('Content-Type', 'application/json')
                            self.send_header('Access-Control-Allow-Origin', '*')
                            self.end_headers()
                            self.wfile.write(json.dumps(response).encode('utf-8'))

                            if permissions.get("superAdmin") is True:
                                logging.info(f"Login successful (Super Admin) for user: {username}")
                            else:
                                logging.info(f"Login successful for user: {username}")
                            return

                self.send_response(401)
                self.send_header('Content-Type', 'application/json')
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(json.dumps({
                    "success": False,
                    "message": "Invalid credentials."
                }).encode('utf-8'))
                logging.warning(f"Invalid credentials for user: {username}")
                return

            except Exception as e:
                logging.error(f"Error processing POST /login: {e}")
                self.send_response(500)
                self.send_header('Content-Type', 'application/json')
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(json.dumps({
                    "success": False,
                    "message": "Internal server error."
                }).encode('utf-8'))
                return
            
        elif parsed_path.path == '/Add_Mod':
            target_server = self.headers.get('X-Target-Server')
            server_name = get_server_name(target_server)
            username = self.headers.get('X-User', 'Unknown')
            
            if not target_server:
                raise ValueError("Missing X-Target-Server header")
            if target_server not in SERVER_URLS:
                raise ValueError(f"Invalid target server: {target_server}")
            try:
                ctype, pdict = cgi.parse_header(self.headers.get('Content-Type'))
                if ctype != 'multipart/form-data':
                    self.send_response(400)
                    self.send_header('Content-Type', 'application/json')
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(json.dumps({
                        "success": False,
                        "message": "Content-Type must be multipart/form-data"
                    }).encode('utf-8'))
                    return

                pdict['boundary'] = bytes(pdict['boundary'], "utf-8")
                pdict['CONTENT-LENGTH'] = int(self.headers.get('Content-Length', 0))
                form = cgi.FieldStorage(
                    fp=self.rfile,
                    headers=self.headers,
                    environ={'REQUEST_METHOD': 'POST'},
                    keep_blank_values=True
                )

                if 'file' not in form:
                    raise ValueError("Missing file field")

                mod_file_field = form['file']
                if not mod_file_field.filename:
                    raise ValueError("No file uploaded")

                filename = mod_file_field.filename
                file_data = mod_file_field.file.read()
                upload_url = SERVER_URLS[target_server]['base'] + '/Mod_Upload'
                logging.info(f"{username} -> Mod_Upload on {server_name}: {filename}")

                files = {
                    'file': (filename, io.BytesIO(file_data))
                }

                forward_response = requests.post(upload_url, files=files)
                forward_response.raise_for_status()
                forward_json = forward_response.json()

                self.send_response(200)
                self.send_header('Content-Type', 'application/json')
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(json.dumps(forward_json).encode('utf-8'))
                logging.info(f"Server -> Mod_Upload forwarded successfully to {server_name}")

            except ValueError as e:
                logging.warning(f"Bad request /Mod_Upload: {e}")
                self.send_response(400)
                self.send_header('Content-Type', 'application/json')
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(json.dumps({
                    "success": False,
                    "message": str(e)
                }).encode('utf-8'))

            except Exception as e:
                logging.error(f"Error processing /Mod_Upload: {e}")
                self.send_response(500)
                self.send_header('Content-Type', 'application/json')
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(json.dumps({
                    "success": False,
                    "message": "Internal server error"
                }).encode('utf-8'))

            return

        else:
            self.send_response(404)
            self.send_header('Content-Type', 'application/json')
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            self.wfile.write(json.dumps({
                "success": False,
                "message": "Unknown POST endpoint."
            }).encode('utf-8'))

    def do_PUT(self):
        content_length = int(self.headers['Content-Length'])
        put_data = self.rfile.read(content_length).decode('utf-8').strip()

        # ==== File System Processing ====
        relay_endpoints = ["/list", "/read", "/write", "/delete", "/rename"]

        if self.path in relay_endpoints:
            target = self.headers.get("X-Target-Server")
            username = self.headers.get("X-User", "Unknown")

            target_key = None
            if target:
                target_lower = target.lower()
                for key in SERVER_URLS.keys():
                    if target_lower in key.lower():
                        target_key = key
                        break

            if not target_key:
                self.send_response(400)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Invalid or missing X-Target-Server header.")
                return

            target_url = f"{SERVER_URLS[target_key]['base']}{self.path}"
            server_name = get_server_name(target_key)

            try:
                response = requests.put(target_url, data=put_data, headers={"Content-Type": "application/json"})
                self.send_response(response.status_code)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.send_header('Content-Type', 'application/json')
                self.end_headers()
                self.wfile.write(response.content)

                if any(p in self.path for p in ["/write", "/delete", "/rename"]):
                    try:
                        json_data = json.loads(put_data if isinstance(put_data, str) else put_data.decode('utf-8'))
                        if self.path == "/rename":
                            file_path = json_data.get('old_path') or 'unknown path'
                        else:
                            file_path = json_data.get('file') or json_data.get('path') or 'unknown path'

                        logging.info(f"{username} -> File system change on {server_name} {self.path} {file_path}")
                    except Exception as e:
                        logging.warning(f"{username} -> Failed to log file system change: {e}")
                return

            except requests.exceptions.RequestException as e:
                self.send_response(502)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(f"Relay error: {str(e)}".encode("utf-8"))
                logging.error(f"Server -> Exception forwarding command to {server_name}: {e}")
                return

        # ==== Endpoint Processing ====
        elif self.path in ('/Start_Server', '/Stop_Server'):
            target_server = self.headers.get('X-Target-Server')
            username = self.headers.get('X-User', 'Unknown')

            if target_server not in SERVER_URLS:
                self.send_response(400)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Invalid or missing X-Target-Server header.")
                return

            ops_port = OPS_PORTS.get(target_server)
            if not ops_port:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"OPS port not configured for target server.")
                return

            srv = config.servers[target_server]
            server_name = srv["name"]
            ip = srv["ip"]

            forward_url = f"http://{ip}:{ops_port}{self.path}"

            try:
                logging.info(f"{username} -> {self.path.lstrip('/')} on {server_name}")

                response = requests.put(
                    forward_url,
                    timeout=10
                )

                self.send_response(response.status_code)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(response.content)

                if response.status_code == 200:
                    logging.info(f"Server -> Command forwarded successfully to {server_name}")
                else:
                    logging.error(
                        f"Server -> {self.path} failed on {server_name} "
                        f"(status {response.status_code})"
                    )

            except requests.exceptions.RequestException as e:
                self.send_response(502)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(f"Relay error: {e}".encode('utf-8'))
                logging.error(f"Server -> Exception forwarding command to {server_name}: {e}")
            return
        
        elif self.path == '/List_Users':
            try:
                with open(FILE_CONTROLPANEL_USERS, 'r') as f:
                    users = json.load(f)
                self.send_response(200)
                self.send_header('Content-Type', 'application/json')
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(json.dumps(users).encode('utf-8'))
            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                logging.error(f"❌ Failed to list users: {e}")
            return

        elif self.path == '/Add_User':
            if '@' in put_data:
                username, password = put_data.split('@', 1)
                self.add_user(username, password)
            else:
                self.send_response(400)
                self.end_headers()
                self.wfile.write(b"Invalid user data format.")
            return

        elif self.path == '/Remove_User':
            if '@' in put_data:
                username, password = put_data.split('@', 1)
                self.remove_user(username, password)
            else:
                self.send_response(400)
                self.end_headers()
                self.wfile.write(b"Invalid user data format.")
            return

        elif self.path == '/Modify_User':
            usernameLog = self.headers.get('X-User', 'Unknown')
            try:
                data = json.loads(put_data)

                username = data.get('username')
                password = data.get('password')
                new_permissions = data.get('permissions')

                if not username or not password or not isinstance(new_permissions, dict):
                    self.send_response(400)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b'Invalid request format.')
                    return

                if not os.path.exists(FILE_CONTROLPANEL_USERS):
                    self.send_response(500)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b'User data file not found.')
                    return

                with open(FILE_CONTROLPANEL_USERS, 'r') as f:
                    users = json.load(f)

                user_found = False
                for user in users:
                    if user['username'] == username and user['password'] == password:
                        user['permissions'] = new_permissions
                        user_found = True
                        break

                if not user_found:
                    self.send_response(404)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b'User not found or password incorrect.')
                    return

                with open(FILE_CONTROLPANEL_USERS, 'w') as f:
                    json.dump(users, f, indent=2)

                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b'Permissions updated successfully.')
                logging.info(f"{usernameLog} -> Updated permissions for: {username}")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                error_msg = f'❌ Server error while modifying user: {e}'
                logging.error(error_msg)
                self.wfile.write(error_msg.encode('utf-8'))
            return

        history_endpoint = FILE_PLAYERHISTORY.get(self.path)
        if history_endpoint:
            try:
                with open(history_endpoint, 'r', encoding='utf-8') as f:
                    history = json.load(f)

                self.send_response(200)
                self.send_header('Content-Type', 'application/json')
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(json.dumps(history).encode('utf-8'))

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(
                    f"Error reading history file: {e}".encode('utf-8')
                )
            return

        elif self.path == '/Add_Mod':
            print("Entered /Add_Mod handler")
            target_server = self.headers.get("X-Target-Server")
            target_action = self.headers.get("X-Action")
            server_name = get_server_name(target_server)
            username = self.headers.get("X-User", "Unknown")

            if target_server not in SERVER_URLS:
                print("Invalid or missing X-Target-Server header")
                self.send_response(400)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Invalid or missing X-Target-Server header.")
                return

            addModURL = SERVER_URLS[target_server]['addmod']
            try:
                logging.info(f"{username} -> Add_Mod on {server_name}: {put_data}")
                response = requests.put(addModURL, data=put_data.encode('utf-8'))

                if response.status_code == 200:
                    self.send_response(200)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Command forwarded successfully.")
                    logging.info(f"Server -> Add_Mod forwarded successfully to {server_name}")
                else:
                    self.send_response(502)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Error forwarding command to server.")
                    logging.error(f"Server -> Error {response.status_code} from {server_name}")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
            return

        elif self.path == '/Server_Logs':
                target_server = self.headers.get('X-Target-Server')
                username = self.headers.get('X-User', 'Unknown')

                server = SERVER_URLS.get(target_server)
                if not server:
                        self.send_response(400)
                        self.send_header('Access-Control-Allow-Origin', '*')
                        self.end_headers()
                        self.wfile.write(b"Missing or invalid 'X-Target-Server' header.")
                        return

                log_file = server["rust_log"]
                server_name = server["name"]

                try:
                        with open(log_file, 'r', encoding='utf-8') as f:
                                lines = f.readlines()

                        recent_lines = lines[-3000:]

                        with open(log_file, 'w', encoding='utf-8') as f:
                                f.writelines(recent_lines)

                        self.send_response(200)
                        self.send_header('Access-Control-Allow-Origin', '*')
                        self.end_headers()
                        self.wfile.write(b"Log file truncated to last 3000 entries.")

                        logging.info(
                                f"{username} -> Trimmed {server_name} logs to latest 3000 entries"
                        )

                except Exception as e:
                        self.send_response(500)
                        self.send_header('Access-Control-Allow-Origin', '*')
                        self.end_headers()

                        error_msg = (
                                f"{username} -> Failed to trim log file for {server_name}: {e}"
                        )
                        self.wfile.write(error_msg.encode('utf-8'))
                        logging.error(error_msg)
                return

        elif self.path == '/Panel_Logs':
            username = self.headers.get('X-User', 'Unknown')
            log_file = FILE_CONTROLPANEL_LOGS

            try:
                with open(log_file, 'r', encoding='utf-8') as f:
                    lines = f.readlines()

                recent_lines = lines[-500:]

                with open(log_file, 'w', encoding='utf-8') as f:
                    f.writelines(recent_lines)

                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Panel log truncated to last 500 entries.")
                logging.info(f"{username} -> Trimmed control panel logs to latest 500 entries")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                error_msg = f"{username} -> Failed to truncate panel logs: {e}"
                self.wfile.write(error_msg.encode('utf-8'))
                logging.error(error_msg)
            return

        elif self.path == '/Update_Panel':
            username = self.headers.get('X-User', 'Unknown')
            logging.info(f"{username} -> Initiated the IRCP update process")

            try:
                start_script("UpdateControlPanel.py")
                time.sleep(2)

                if not is_script_running("UpdateControlPanel"):
                    raise RuntimeError("UpdateControlPanel.py handler didn't start or exited unexpectedly")

                logging.info("Control Panel update process has started")

                for key, srv in config.servers.items():
                    port = RCON_PORTS.get(key)
                    ip = srv["ip"]
                    name = srv["name"]

                    result = start_rust_updater(ip, port)

                    if result == 200:
                        logging.info(f"{name} update process has started")
                    else:
                        logging.error(
                            f"{name} failed to start the update process"
                        )
                        raise RuntimeError("UpdateRustServer.py handler didn't start or exited unexpectedly")

                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Panel update process started.")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()

                error_msg = f"Failed to complete the update process: {e}"
                self.wfile.write(b"Panel update process failed.")
                logging.error(error_msg)
            return

        elif self.path == '/Remove_Delete':
            target_server = self.headers.get('X-Target-Server', '')
            server_name = get_server_name(target_server)
            username = self.headers.get('X-User', 'Unknown')

            if target_server == 'Server_One':
                history_file = FILE_PLAYERHISTORY_ONE
            elif target_server == 'Server_Two':
                history_file = FILE_PLAYERHISTORY_TWO
            elif target_server == 'Server_Three':
                history_file = FILE_PLAYERHISTORY_THREE
            elif target_server == 'Server_Four':
                history_file = FILE_PLAYERHISTORY_FOUR
            else:
                self.send_response(400)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Invalid X-Target-Server header.")
                return

            try:
                data = json.loads(put_data)
                player_name = data.get("player_name")

                if not player_name:
                    self.send_response(400)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Missing player_name in request body.")
                    return

                with open(history_file, 'r', encoding='utf-8') as f:
                    history = json.load(f)

                original_len = len(history)
                updated_history = [p for p in history if p.get("name", "").lower() != player_name.lower()]

                if len(updated_history) == original_len:
                    self.send_response(404)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Player not found in history.")
                    return

                with open(history_file, 'w', encoding='utf-8') as f:
                    json.dump(updated_history, f, indent=2)

                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Player history deleted successfully.")
                logging.info(f"{username} -> Deleted play history for player: {player_name} on {server_name}")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                error_msg = f"Error deleting player from history: {e}"
                self.wfile.write(error_msg.encode('utf-8'))
                logging.error(f"{username} -> Failed to delete: {error_msg}")
            return

        elif self.path == '/Clear':
            target_server = self.headers.get('X-Target-Server', '')
            server_name = get_server_name(target_server)
            username = self.headers.get('X-User', 'Unknown')

            if target_server == 'Server_One':
                history_file = FILE_PLAYERHISTORY_ONE
            elif target_server == 'Server_Two':
                history_file = FILE_PLAYERHISTORY_TWO
            elif target_server == 'Server_Three':
                history_file = FILE_PLAYERHISTORY_THREE
            elif target_server == 'Server_Four':
                history_file = FILE_PLAYERHISTORY_FOUR
            else:
                self.send_response(400)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Invalid X-Target-Server header.")
                return

            try:
                data = json.loads(put_data)
                player_name = data.get("player_name")

                if not player_name:
                    self.send_response(400)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Missing player_name in request body.")
                    return

                with open(history_file, 'r', encoding='utf-8') as f:
                    history = json.load(f)

                found = False
                for player in history:
                    if player.get("name", "").lower() == player_name.lower():
                        player["time_seconds"] = 0
                        player["time"] = "00:00:00"
                        player["last_active"] = "Last-Active-Unknown"
                        found = True
                        break

                if not found:
                    self.send_response(404)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Player not found in history.")
                    return

                with open(history_file, 'w', encoding='utf-8') as f:
                    json.dump(history, f, indent=2)

                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Player history cleared successfully.")
                logging.info(f"{username} -> Reset play history for player: {player_name} on {server_name}")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                error_msg = f"Error clearing player history: {e}"
                self.wfile.write(error_msg.encode('utf-8'))
                logging.error(f"{username} -> Failed to reset history: {error_msg}")
            return

        elif self.path == '/Hide':
            target_server = self.headers.get('X-Target-Server', '')
            server_name = get_server_name(target_server)
            username = self.headers.get('X-User', 'Unknown')

            if target_server == 'Server_One':
                history_file = FILE_PLAYERHISTORY_ONE
            elif target_server == 'Server_Two':
                history_file = FILE_PLAYERHISTORY_TWO
            elif target_server == 'Server_Three':
                history_file = FILE_PLAYERHISTORY_THREE
            elif target_server == 'Server_Four':
                history_file = FILE_PLAYERHISTORY_FOUR
            else:
                self.send_response(400)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Invalid X-Target-Server header.")
                return

            try:
                data = json.loads(put_data)
                player_name = data.get("player_name")

                if not player_name:
                    self.send_response(400)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Missing player_name in request body.")
                    return

                with open(history_file, 'r', encoding='utf-8') as f:
                    history = json.load(f)

                found = False
                for player in history:
                    if player.get("name", "").lower() == player_name.lower():
                        player["hidden"] = True
                        found = True
                        break

                if not found:
                    self.send_response(404)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Player not found in history.")
                    return

                with open(history_file, 'w', encoding='utf-8') as f:
                    json.dump(history, f, indent=2)

                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Player successfully removed from history view.")
                logging.info(f"{username} -> Set play history for: {player_name} as hidden on {server_name}")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                error_msg = f"Error hiding player from history: {e}"
                self.wfile.write(error_msg.encode('utf-8'))
                logging.error(f"{username} -> Failed to set player as hidden: {error_msg}")
            return

        elif self.path == '/Unhide':
            target_server = self.headers.get('X-Target-Server', '')
            server_name = get_server_name(target_server)
            username = self.headers.get('X-User', 'Unknown')

            if target_server == 'Server_One':
                history_file = FILE_PLAYERHISTORY_ONE
            elif target_server == 'Server_Two':
                history_file = FILE_PLAYERHISTORY_TWO
            elif target_server == 'Server_Three':
                history_file = FILE_PLAYERHISTORY_THREE
            elif target_server == 'Server_Four':
                history_file = FILE_PLAYERHISTORY_FOUR
            else:
                self.send_response(400)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Invalid X-Target-Server header.")
                return

            try:
                data = json.loads(put_data)
                player_name = data.get("player_name")

                if not player_name:
                    self.send_response(400)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Missing player_name in request body.")
                    return

                with open(history_file, 'r', encoding='utf-8') as f:
                    history = json.load(f)

                found = False
                for player in history:
                    if player.get("name", "").lower() == player_name.lower():
                        player["hidden"] = False
                        found = True
                        break

                if not found:
                    self.send_response(404)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"Player not found in history.")
                    return

                with open(history_file, 'w', encoding='utf-8') as f:
                    json.dump(history, f, indent=2)

                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Player successfully unhidden in history view.")
                logging.info(f"{username} -> Set play history for: {player_name} as unhidden on {server_name}")

            except Exception as e:
                self.send_response(500)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                error_msg = f"Error unhiding player from history: {e}"
                self.wfile.write(error_msg.encode('utf-8'))
                logging.error(f"{username} -> Failed to set player as unhidden: {error_msg}")
            return

        # ==== COMMAND PROCESSING ====
        target_server = self.headers.get('X-Target-Server')
        server_name = get_server_name(target_server)
        username = self.headers.get('X-User', 'Unknown')
        if target_server not in SERVER_URLS:
            self.send_response(400)
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            self.wfile.write(b"Missing or invalid 'X-Target-Server' header.")
            logging.warning("Missing or invalid target server header.")
            return

        fastRestartURL = SERVER_URLS[target_server]['fast']
        delayedRestartURL = SERVER_URLS[target_server]['delayed']
        customCommandURL = SERVER_URLS[target_server]['custom']
        fullWipeURL = SERVER_URLS[target_server]['fullwipe']
        halfWipeURL = SERVER_URLS[target_server]['halfwipe']
        sayURL = SERVER_URLS[target_server]['say']

        try:
            if put_data == "Fast_Restart":
                logging.info(f"{username} -> Fast_Restart on {server_name}")
                response = requests.put(fastRestartURL)
            elif put_data == "Delayed_Restart":
                logging.info(f"{username} -> Delayed_Restart on {server_name}")
                response = requests.put(delayedRestartURL)
            elif put_data == "Full_Wipe":
                logging.info(f"{username} -> Full_Wipe on {server_name}")
                response = requests.put(fullWipeURL)
            elif put_data == "Half_Wipe":
                logging.info(f"{username} -> Half_Wipe on {server_name}")
                response = requests.put(halfWipeURL)
            elif put_data == "Say":
                logging.info(f"{username} -> Say on {server_name}")
                response = requests.put(sayURL)
            else:
                logging.info(f"{username} -> Command on {server_name}: {put_data}")
                response = requests.put(customCommandURL, data=put_data.encode('utf-8'))

            if response.status_code == 200:
                self.send_response(200)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Command forwarded successfully.")
                logging.info(f"Server -> Command forwarded successfully to {server_name}")
            else:
                self.send_response(502)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"Error forwarding command to server.")
                logging.error(f"Server -> Error {response.status_code} from {server_name}")

        except requests.exceptions.RequestException as e:
            self.send_response(500)
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            self.wfile.write(b"Exception during forwarding.")
            logging.error(f"Server -> Exception forwarding command to {server_name}: {e}")

    # ==== Helper functions ====       
    def add_user(self, username, password):
        json_file_path = FILE_CONTROLPANEL_USERS
        usernameLog = self.headers.get('X-User', 'Unknown')
        os.makedirs(os.path.dirname(json_file_path), exist_ok=True)

        default_permissions = {
            "owner": False
        }

        try:
            if os.path.exists(json_file_path):
                with open(json_file_path, 'r') as f:
                    try:
                        users = json.load(f)
                    except json.JSONDecodeError:
                        users = []
            else:
                users = []

            for user in users:
                if user["username"] == username:
                    self.send_response(409)
                    self.send_header('Access-Control-Allow-Origin', '*')
                    self.end_headers()
                    self.wfile.write(b"User already exists.")
                    return

            users.append({
                "username": username,
                "password": password,
                "permissions": default_permissions
            })

            with open(json_file_path, 'w') as f:
                json.dump(users, f, indent=2)

            logging.info(f"{usernameLog} -> Added user: {username}")
            self.send_response(200)
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            self.wfile.write(b"User added successfully.")

        except Exception as e:
            logging.error(f"Error adding user: {e}")
            self.send_response(500)
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            self.wfile.write(b"Failed to add user.")

    def remove_user(self, username, password):
        json_file_path = FILE_CONTROLPANEL_USERS
        usernameLog = self.headers.get('X-User', 'Unknown')
        try:
            if not os.path.exists(json_file_path):
                self.send_response(404)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"User file not found.")
                return

            with open(json_file_path, 'r') as f:
                try:
                    users = json.load(f)
                except json.JSONDecodeError:
                    users = []

            # Filter out the user
            new_users = [user for user in users if not (user['username'] == username and user['password'] == password)]

            if len(new_users) == len(users):
                self.send_response(404)
                self.send_header('Access-Control-Allow-Origin', '*')
                self.end_headers()
                self.wfile.write(b"User not found.")
                return

            with open(json_file_path, 'w') as f:
                json.dump(new_users, f, indent=4)

            logging.info(f"{usernameLog} -> Removed user: {username}")
            self.send_response(200)
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            self.wfile.write(b"User removed successfully.")

        except Exception as e:
            logging.error(f"Error removing user: {e}")
            self.send_response(500)
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            self.wfile.write(b"Failed to remove user.")

def get_server_name(server_key):
    return SERVER_URLS.get(server_key, {}).get("name", server_key)

def start_script(script_name):
    if platform.system() == "Windows":
        si = subprocess.STARTUPINFO()
        si.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        si.wShowWindow = 2

        subprocess.Popen(
            [sys.executable, script_name],
            startupinfo=si,
            creationflags=subprocess.CREATE_NEW_CONSOLE
        )
    else:
        subprocess.Popen([sys.executable, script_name])

def is_script_running(script_name):
    system = platform.system()
    try:
        if system == "Windows":
            cmd = ['wmic', 'process', 'get', 'CommandLine']
        else:
            cmd = ['ps', 'aux']

        output = subprocess.check_output(cmd, text=True, errors='ignore')
        return any(script_name in line and 'python' in line for line in output.splitlines())
    except Exception:
        return False

def start_rust_updater(ip, port, timeout=10):
    conn = http.client.HTTPConnection(ip, port, timeout=timeout)
    try:
        conn.request("PUT", "/Update_Rust")
        response = conn.getresponse()
        status = response.status
        response.read()
        return status
    except (socket.timeout, ConnectionError, OSError) as e:
        return f"error:{e}"
    finally:
        conn.close()

if __name__ == "__main__":
    with socketserver.TCPServer(("", PORT), BackendRequestHandler) as httpd:
        print(f"Serving on port {PORT}")
        logging.info(f"Server started on port {PORT}")
        httpd.serve_forever()
        
