from http.server import SimpleHTTPRequestHandler, HTTPServer
import urllib.request
import importlib.metadata
import socket
import subprocess
import sys
import platform
import os
import threading
import time
import ctypes
import re
import ast

ctypes.windll.kernel32.SetConsoleTitleW("ControlPanelServer")
BASE_DIR = os.path.dirname(os.path.abspath(__file__))

# Network Port
PORT = 28000

config_py = os.path.join(BASE_DIR, "config.py")
config_js = os.path.join(BASE_DIR, "config.js")
users_json = os.path.join(BASE_DIR, "panelusers.json")

def clear_console():
    os.system('cls' if os.name == 'nt' else 'clear')
    
def print_banner():
    print(r"""
================================================================================
================================================================================
====                                                                        ====
====    ██╗██████╗  ██████╗██████╗                                          ====
====    ██║██╔══██╗██╔════╝██╔══██╗                                         ====
====    ██║██████╔╝██║     ██████╔╝   Integrated Rust Control Panel         ====
====    ██║██╔══██╗██║     ██╔═══╝    Brought to you by Im_just_a_Pixel     ====
====    ██║██║  ██║╚██████╗██║                                              ====
====    ╚═╝╚═╝  ╚═╝ ╚═════╝╚═╝                                              ====
====                                                                        ====
================================================================================
================================================================================
""")

def oops_banner_init():
    print(r"""        ██████████████████      
      ██████████████████████    
    ████████          ████████  
   ███████  (o)    (o)   ██████  
  ███████                 ██████ 
  ██████       ▄▄▄▄       ██████ 
  ██████      █░░░░█       █████           
  ██████       ▀▀▀▀        █████ 
  ██████                   █████
  ███████                 ██████
   ████████     ░ ░     ███████
    █████████████████████████
      █████████████████████
        █████████████████
          ████     ████
        ███████   ███████
     ███████████ ███████████
    ███    ███████████    ███
    ██     ███████████     ██
    ██      █████████      ██
    █       █████████       █
             ███████
            ███   ███
            ███   ███
             ██   ██
             ██   ██
              █   █
              █   █
""")

def oops_banner_hey():
    print(r"""        ██████████████████      
      ██████████████████████    
    ████████          ████████  
   ███████  (o)    (o)   ██████            ┌────────────────┐
  ███████                 ██████           │                │
  ██████       ▄▄▄▄       ██████           │  Hey there!    │
  ██████      █░░░░█       █████   ◄───────┤                │           
  ██████       ▀▀▀▀        █████           └────────────────┘
  ██████                   █████
  ███████                 ██████
   ████████     ░ ░     ███████
    █████████████████████████
      █████████████████████
        █████████████████
          ████     ████
        ███████   ███████
     ███████████ ███████████
    ███    ███████████    ███
    ██     ███████████     ██
    ██      █████████      ██
    █       █████████       █
             ███████
            ███   ███
            ███   ███
             ██   ██
             ██   ██
              █   █
              █   █
""")

def oops_banner_check():
    print(r"""        ██████████████████      
      ██████████████████████    
    ████████          ████████  
   ███████  (o)    (o)   ██████            ┌──────────────────────────────────────────────┐
  ███████                 ██████           │  Give me a moment while I check out          │
  ██████       ▄▄▄▄       ██████           │  your configs.                               │
  ██████      █░░░░█       █████   ◄───────┤                                              │           
  ██████       ▀▀▀▀        █████           └──────────────────────────────────────────────┘
  ██████                   █████
  ███████                 ██████
   ████████     ░ ░     ███████
    █████████████████████████
      █████████████████████
        █████████████████
          ████     ████
        ███████   ███████
     ███████████ ███████████
    ███    ███████████    ███
    ██     ███████████     ██
    ██      █████████      ██
    █       █████████       █
             ███████
            ███   ███
            ███   ███
             ██   ██
             ██   ██
              █   █
              █   █
""")

def oops_banner_ok():
    print(r"""        ██████████████████      
      ██████████████████████    
    ████████          ████████  
   ███████  (o)    (o)   ██████            ┌──────────────────────────────────────────────┐
  ███████                 ██████           │  Looks like everything is ready to go.       │
  ██████       ▄▄▄▄       ██████           │  Lets start your control panel.              │
  ██████      █░░░░█       █████   ◄───────┤                                              │           
  ██████       ▀▀▀▀        █████           └──────────────────────────────────────────────┘
  ██████                   █████
  ███████                 ██████
   ████████     ░ ░     ███████
    █████████████████████████
      █████████████████████
        █████████████████
          ████     ████
        ███████   ███████
     ███████████ ███████████
    ███    ███████████    ███
    ██     ███████████     ██
    ██      █████████      ██
    █       █████████       █
             ███████
            ███   ███
            ███   ███
             ██   ██
             ██   ██
              █   █
              █   █
""")

def oops_banner_one():
    print(r"""        ██████████████████      
      ██████████████████████    
    ████████          ████████  
   ███████  (o)    (o)   ██████            ┌──────────────────────────────────────────────┐
  ███████                 ██████           │  OOPS! Looks like there are some             │
  ██████       ▄▄▄▄       ██████           │  configuration details missing.              │
  ██████      █░░░░█       █████   ◄───────┤                                              │           
  ██████       ▀▀▀▀        █████           └──────────────────────────────────────────────┘
  ██████                   █████
  ███████                 ██████
   ████████     ░ ░     ███████
    █████████████████████████
      █████████████████████
        █████████████████
          ████     ████
        ███████   ███████
     ███████████ ███████████
    ███    ███████████    ███
    ██     ███████████     ██
    ██      █████████      ██
    █       █████████       █
             ███████
            ███   ███
            ███   ███
             ██   ██
             ██   ██
              █   █
              █   █
""")

def oops_banner_two():
    print(r"""        ██████████████████      
      ██████████████████████    
    ████████          ████████  
   ███████  (o)    (o)   ██████            ┌──────────────────────────────────────────────┐
  ███████                 ██████           │  OOPS! Looks like there are some             │
  ██████       ▄▄▄▄       ██████           │  configuration details missing.              │
  ██████      █░░░░█       █████   ◄───────┤                                              │           
  ██████       ▀▀▀▀        █████           │  I'll start the control panel configurator   │
  ██████                   █████           │  for you.                                    │
  ███████                 ██████           └──────────────────────────────────────────────┘
   ████████     ░ ░     ███████
    █████████████████████████
      █████████████████████
        █████████████████
          ████     ████
        ███████   ███████
     ███████████ ███████████
    ███    ███████████    ███
    ██     ███████████     ██
    ██      █████████      ██
    █       █████████       █
             ███████
            ███   ███
            ███   ███
             ██   ██
             ██   ██
              █   █
              █   █
""")

def create_required_files():
    required_files = {
        "rustserverone.log": "",
        "rustservertwo.log": "",
        "rustserverthree.log": "",
        "rustserverfour.log": "",
        "panellogs.log": "",
        "historyserverone.json": "{}",
        "historyservertwo.json": "{}",
        "historyserverthree.json": "{}",
        "historyserverfour.json": "{}",
    }

    for filename, default_content in required_files.items():
        path = os.path.join(BASE_DIR, filename)
        if not os.path.exists(path):
            with open(path, "w", encoding="utf-8") as f:
                f.write(default_content)

def install_package(package_name, exact_version=None):
    try:
        installed_version = importlib.metadata.version(package_name)
        
        if exact_version:
            if installed_version == exact_version:
                return True
            print(f"[Dependency] {package_name} {installed_version} found (expected {exact_version}) — fixing...")
        else:
            return True

    except importlib.metadata.PackageNotFoundError:
        if exact_version:
            print(f"[Dependency] {package_name} not found — installing {exact_version}")
        else:
            print(f"[Dependency] {package_name} not found — installing latest version")

    cmd = [sys.executable, "-m", "pip", "install"]
    if exact_version:
        cmd.append(f"{package_name}=={exact_version}")
    else:
        cmd.append(package_name)

    subprocess.check_call(cmd)
    return True

def run_configurator():
    oops_banner_one()
    time.sleep(6)
    clear_console()
    oops_banner_two()
    time.sleep(6)
    clear_console()

    result = subprocess.run(
        [sys.executable, "ConfigureControlPanel.py"],
        check=False
    )

    if result.returncode != 0:
        print("Configuration failed or was cancelled.")
        sys.exit(1)

    clear_console()

def is_script_running(script_name):
    system = platform.system()
    try:
        if system == "Windows":
            cmd = ['wmic', 'process', 'get', 'CommandLine']
        else:
            cmd = ['ps', 'aux']

        output = subprocess.check_output(cmd, text=True, errors='ignore')
        return any(script_name in line and 'python' in line for line in output.splitlines())
    except Exception:
        return False

def start_script(script_name):
    if platform.system() == "Windows":
        si = subprocess.STARTUPINFO()
        si.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        si.wShowWindow = 2

        subprocess.Popen(
            [sys.executable, script_name],
            startupinfo=si,
            creationflags=subprocess.CREATE_NEW_CONSOLE
        )
    else:
        subprocess.Popen([sys.executable, script_name])

def validate_config_js(path):
    try:
        with open(path, "r", encoding="utf-8") as f:
            content = f.read()
    except Exception:
        return False

    cp_regex = r'^var controlPanelName = "[^"]+";$'
    server_regex = r'^var (serverOneName|serverTwoName|serverThreeName|serverFourName) = "[^"]+";$'

    server_vars = []
    seen_servers = set()
    seen_control_panel = False

    for line in content.splitlines():
        line = line.strip()

        if not line or line.startswith("//"):
            continue

        if not line.startswith("var"):
            return False

        if line.startswith("var controlPanelName"):
            if seen_control_panel:
                return False
            if not re.match(cp_regex, line):
                return False
            seen_control_panel = True
            continue

        match = re.match(server_regex, line)
        if not match:
            return False

        server_name = match.group(1)
        if server_name in seen_servers:
            return False

        seen_servers.add(server_name)
        server_vars.append(server_name)

    if not (1 <= len(server_vars) <= 4):
        return False

    if not seen_control_panel:
        return False

    return True

def validate_config_py(path):
    try:
        with open(path, "r", encoding="utf-8") as f:
            tree = ast.parse(f.read(), filename=path)
    except Exception:
        return False

    servers_node = None

    for node in tree.body:
        if isinstance(node, ast.Assign):
            for target in node.targets:
                if isinstance(target, ast.Name) and target.id == "servers":
                    servers_node = node.value

    if servers_node is None or not isinstance(servers_node, ast.Dict):
        return False

    server_count = len(servers_node.keys)
    if not (1 <= server_count <= 4):
        return False

    valid_keys = {"Server_One", "Server_Two", "Server_Three", "Server_Four"}

    seen_keys = set()

    for key, value in zip(servers_node.keys, servers_node.values):
        if not isinstance(key, ast.Constant) or key.value not in valid_keys:
            return False

        if key.value in seen_keys:
            return False
        seen_keys.add(key.value)

        if not isinstance(value, ast.Dict):
            return False

        fields = {}
        for k, v in zip(value.keys, value.values):
            if not isinstance(k, ast.Constant):
                return False
            fields[k.value] = v

        if not {"ip", "name"}.issubset(fields):
            return False

        for field in ("ip", "name"):
            v = fields[field]
            if not isinstance(v, ast.Constant):
                return False
            if not isinstance(v.value, str):
                return False
            if not v.value.strip():  # <-- THIS rejects empty or whitespace-only strings
                return False

    return True

def configs_are_valid():
    return (
        validate_config_js(config_js)
        and validate_config_py(config_py)
    )

def get_lan_ip():
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    try:
        s.connect(("8.8.8.8", 80))
        return s.getsockname()[0]
    except Exception:
        return "127.0.0.1"
    finally:
        s.close()

def get_wan_ip():
    try:
        with urllib.request.urlopen("https://api.ipify.org", timeout=8) as r:
            return r.read().decode().strip()
    except Exception:
        return None

def check_tcp(ip, port, timeout=2):
    try:
        with socket.create_connection((ip, port), timeout=timeout):
            return True
    except Exception:
        return False

class Handler(SimpleHTTPRequestHandler):
    def do_GET(self):
        if self.path == "/":
            self.path = "/controlpanel.html"
        return super().do_GET()

# ==== Main Loop ====
server = HTTPServer(("0.0.0.0", PORT), Handler)

create_required_files()

install_package("psutil", "7.1.3")
install_package("requests", "2.32.5")
install_package("watchdog", "6.0.0")
install_package("websockets", "15.0.1")
install_package("urllib3", "2.6.2")
install_package("charset-normalizer", "3.4.4")
install_package("certifi")
install_package("idna")

time.sleep(1)
oops_banner_init()
time.sleep(1)
clear_console()
oops_banner_hey()
time.sleep(3)
clear_console()
oops_banner_init()
time.sleep(1)
clear_console()
oops_banner_check()
time.sleep(5)
clear_console()
oops_banner_init()
time.sleep(1)
clear_console()

if not os.path.exists(config_py) or not os.path.exists(config_js) or not os.path.exists(users_json):
    run_configurator()
elif not configs_are_valid():
    run_configurator()
else:
    oops_banner_ok()
    time.sleep(5)
    clear_console()

print("Checking dependencies...")

time.sleep(2)
if not is_script_running("BackendListener"):
    start_script("BackendListener.py")
    print("Starting BackendListener...")
else:
    print("BackendListener is running.")
    
time.sleep(1)
if not is_script_running("BackendLogs"):
    start_script("BackendLogs.py")
    print("Starting BackendLogs...")
else:
    print("BackendLogs is running.")
    
time.sleep(1)
print("Checking Server reachability...")

lan_ip = get_lan_ip()
wan_ip = get_wan_ip()

lan_ok = check_tcp(lan_ip, PORT)
wan_ok = check_tcp(wan_ip, PORT) if wan_ip else False

time.sleep(1)
clear_console()
print_banner()
print("Control Panel Server Started")
print(f"LAN : http://{lan_ip}:{PORT} ({'Reachable' if lan_ok else 'Not reachable'})")

if wan_ip:
    if wan_ok:
        print(f"WAN : http://{wan_ip}:{PORT} (Reachable)")
    else:
        print(f"WAN : http://{wan_ip}:{PORT} (Router port forwarding misconfigured or blocked by CGNAT)")
else:
    print(f"WAN : http://{wan_ip}:{PORT} (Could not test server reachability)")

print()
print("--------------------------------------------------------------------------------")

server.serve_forever()

